<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Product;
use App\Models\Notification;
use App\Models\EmployeeNotification;
use App\Models\Log;
use Carbon\Carbon;
use App\Models\Message;
use Sheets;
use DateTime;
use Carbon\CarbonPeriod;
use App\Models\Metier;
use App\Models\Chantier;
use App\Models\Tache;
use App\Models\Famille;
use App\Models\Lieu;
use App\Models\Element;
use App\Models\Materiel;
use App\Models\MaterielSalarie;
use App\Models\Absence;
use App\Models\Event;
use DateInterval;
use DatePeriod;
use App\Models\StockMovement;
use App\Models\Holiday;

use App\Models\MaterielRequest;
use App\Models\ElementRequest;

class AdminController extends Controller
{


    public function dashboard(Request $request)
    {
         try {
            $today = now();
            $startOfYear = $today->copy()->startOfYear();
            $endOfYear = $today->copy()->endOfYear();
    
            // Basic KPIs
            $totalProjects = Chantier::count();
            $activeProjects = Chantier::where('date_de_debut', '<=', $today)
                ->where('date_de_fin', '>=', $today)
                ->count();
            
            // Tasks Statistics
            $totalTasks = Tache::count();
            $completedTasks = Tache::where('status', 'completed')->count();
            $taskCompletionRate = $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100, 1) : 0;
            
            // Employee Statistics
            $employees = User::where('role', 'employee')->get();
            $totalEmployees = $employees->count();
            
            // Monthly Project Distribution
            $monthlyProjects = Chantier::selectRaw('MONTH(date_de_debut) as month, COUNT(*) as count')
                ->whereYear('date_de_debut', $today->year)
                ->groupBy('month')
                ->orderBy('month')
                ->get()
                ->pluck('count', 'month')
                ->toArray();
            
            // Fill in missing months with 0
            for ($i = 1; $i <= 12; $i++) {
                if (!isset($monthlyProjects[$i])) {
                    $monthlyProjects[$i] = 0;
                }
            }
            ksort($monthlyProjects);
    
            // Task Status Distribution
            $taskStatusDistribution = Tache::selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status')
                ->toArray();
    
            // Employee Workload (Tasks per Employee)
            $employeeWorkload = collect();
            foreach ($employees as $employee) {
                $activeTasks = Tache::where('status', '!=', 'Terminée')
                    ->whereRaw('JSON_CONTAINS(employees, ?)', ['"' . $employee->id . '"'])
                    ->count();
                
                $employeeWorkload->push([
                    'name' => $employee->name,
                    'tasks_count' => $activeTasks
                ]);
            }
    
            // Project Timeline
            $upcomingProjects = Chantier::where('date_de_debut', '>', $today)
                ->orderBy('date_de_debut')
                ->limit(5)
                ->get()
                ->map(function($project) {
                    return [
                        'nom' => $project->nom,
                        'debut' => $project->date_de_debut,
                        'fin' => $project->date_de_fin,
                        'duree' => $project->nbjours
                    ];
                });
    
            // Task Completion Trend (Last 7 days)
            $taskCompletionTrend = [];
            for ($i = 6; $i >= 0; $i--) {
                $date = $today->copy()->subDays($i)->format('Y-m-d');
                $count = Tache::where('status', 'completed')
                    ->whereDate('updated_at', $date)
                    ->count();
                $taskCompletionTrend[$date] = $count;
            }
    
            // Get ongoing projects with progress
            $ongoingProjects = Chantier::with('taches')
                ->where('date_de_debut', '<=', $today)
                ->where('date_de_fin', '>=', $today)
                ->get()
                ->map(function($project) {
                    $totalTasks = $project->taches->count();
                    $completedTasks = $project->taches->where('status', 'completed')->count();
                    $progress = $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100, 1) : 0;
                    return [
                        'name' => $project->nom,
                        'progress' => $progress,
                        'total_tasks' => $totalTasks,
                        'completed_tasks' => $completedTasks
                    ];
                });
    
            // Get elements with low stock
            $lowStockElements = Element::whereNotNull('stock_alert')
                ->whereRaw('quantity < stock_alert')
                ->get();
                
            $chantiers = Chantier::all();
            
            $totalChantiers = $chantiers->count();
            $onTimeChantiers = $chantiers->filter(function ($chantier) {
                if (empty($chantier->date_fin_reelle)) {
                    return false; // Exclude chantiers without a completion date
                }
            
                return Carbon::parse($chantier->date_fin_reelle) <= Carbon::parse($chantier->date_de_fin);
            })->count();
            
            $lateChantiers = $totalChantiers - $onTimeChantiers;
    
            return view('admin.dashboard', compact(
                'totalProjects',
                'activeProjects',
                'totalTasks',
                'completedTasks',
                'taskCompletionRate',
                'totalEmployees',
                'ongoingProjects',
                'monthlyProjects',
                'taskStatusDistribution',
                'employeeWorkload',
                'upcomingProjects',
                'taskCompletionTrend',
                'lowStockElements',
                
                'totalChantiers', 'onTimeChantiers', 'lateChantiers'
            ));
            
        } catch (\Exception $e) {
            
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }



    public function metiers(){
        $metiers = Metier::orderBy('id','desc')->get();;
        return view('admin.metiers',compact('metiers'));
    }
    public function getMetiers()
    {
        return Metier::all(); // Return all metiers as JSON for AJAX requests
    }


    public function getMetiersArray()
    {
        $metiers = Metier::get(); // Return all metiers as JSON for AJAX requests

        $formattedMetiers = [];

        foreach ($metiers as $metier) {

            $formattedMetier = [
                'id' => $metier->id,
                'nom' => $metier->nom,
                'date' => $metier->date,
            ];
            $formattedMetiers[] = $formattedMetier;
        }

        return response()->json($formattedMetiers);
    }
    public function addMetier(){
        
        $employees = User::where('role', 'employee')->get();
        return view('admin.add-metier', compact("employees"));
    }

    public function insertMetier(Request $request){
        $request->validate([
            'nom'=>'required',
        ]);
         
        try {    
            DB::beginTransaction();
            $metier = new Metier();
            $metier->nom = $request->nom;
            $metier->date = Date('Y-m-d');
            if(isset($request->employees)){
                $metier->employees = json_encode($request->employees);
            }
            $metier->save();
            DB::commit();
            return redirect()->route('admin.metiers')->with('success','Metier Ajouté !');
         

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'An error occurred while importing order');
        }
    }
    public function editMetier(Request $request){
        $metier = Metier::find($request->id);
        if(isset($metier)){
            $metiers = Metier::get();
            $employees = User::where('role', 'employee')->get();
            
            return view("admin.edit-metier",compact("metier","employees"));
        }else{
            return redirect()->route("admin.metiers")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateMetier(Request $request){
        $request->validate([
            'metier' => 'required',
            'nom' => 'required',
        ]);
        try {
            
            $metier = Metier::find($request->metier);
            if(isset($metier)){
                
                DB::beginTransaction();
                
                $metier->nom = $request->nom;
                // $metier->date = Date('Y-m-d');
                if(isset($request->employees)){
                    $metier->employees = json_encode($request->employees);
                }else{
                    $metier->employees = [];
                }
                $metier->update();
                
                DB::commit();
                return redirect()->route('admin.metiers')->with("success","Success! Therapy Updated Successfuly!");
            }else{
                return redirect()->route("admin.metiers")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteMetier(Request $request){
        try {
            $metier = Metier::find($request->id);
            if(isset($metier)){
                
                DB::beginTransaction();
                $metier->delete();
                
                DB::commit();
                return redirect()->route('admin.metiers')->with("success","Success! Therapy Deleted Successfuly!");
                
            }else{
                return redirect()->route("admin.metiers")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }





    
    public function chantiers(Request $request){
        $chantiers = Chantier::orderBy('id','desc')->get();
        return view('admin.chantiers',compact('chantiers'));
    }

    public function addChantier(Request $request){
        return view('admin.add-chantier');
    }
    public function insertChantier(Request $request){
        $request->validate([
            'nom' =>'required',
            'adresse' =>'required',
            'date_de_debut' =>'required',
            'date_de_fin' =>'required',
            'photos.*' => 'max:2048',
        ]);
        try {
            $errors = "";
    
            if ($errors) {
                return redirect()->back()->with("error", $errors);
            }
            else{
                DB::beginTransaction();
                $chantier = new Chantier();
                $chantier->nom = $request->nom;
                $chantier->adresse = $request->adresse;
                $chantier->date = Date('Y-m-d');
                $chantier->date_de_debut = $request->date_de_debut;
                $chantier->date_de_fin = $request->date_de_fin;
                // $chantier->date_fin_reelle = $request->date_fin_reelle;
    
                $chantier->nbjours = $this->calculateWorkingDays($request->date_de_debut, $request->date_de_fin);
    
    
                if ($request->hasFile('photos')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photos') as $file) {
                        $file_name = "Chantier-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $chantier->photo = json_encode($fileNames); // Save filenames as JSON
                }
    
                $chantier->save();
                
                DB::commit();
        
                return redirect()->route('admin.chantiers')->with('success','Success! Chantier Ajouté.');
            }


        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }
    public function editChantier(Request $request){
        $chantier = Chantier::find($request->id);
        if(isset($chantier)){
            return view("admin.edit-chantier", compact("chantier"));
        }else{
            return redirect()->route("admin.chantiers")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateChantier(Request $request){
        $request->validate([
            'id' => 'required',
            'nom' =>'required',
            'adresse' =>'required',
            'photos.*' => 'max:2048',
        ]);

    try {
            $chantier = Chantier::find($request->id);
    
            if(isset($chantier)){
                DB::beginTransaction();
                
                $chantier->nom = $request->nom;
                $chantier->adresse = $request->adresse;
                // $chantier->date = Date('Y-m-d');
                $chantier->date_de_debut = $request->date_de_debut;
                $chantier->date_de_fin = $request->date_de_fin;

                $chantier->date_fin_reelle = $request->date_fin_reelle;
                
                
                $chantier->nbjours = $this->calculateWorkingDays($request->date_de_debut, $request->date_de_fin);
                
                
                if(isset($request->date_fin_reelle)){
                    $chantier->date_fin_reelle = $request->date_fin_reelle;
                    $chantier->nbjours_reelle = $this->calculateWorkingDays($request->date_de_debut, $request->date_fin_reelle);
                }
    
                
    
                if ($request->hasFile('photos')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photos') as $file) {
                        $file_name = "Chantier-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $chantier->photo = json_encode($fileNames); // Save filenames as JSON
                }
            
                $chantier->save();
                
                DB::commit();
    
                return redirect()->route("admin.chantiers")->with("success",'Success! Chantier Modifié.');
            } else{
                return redirect()->route("admin.chantiers")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteChantier(Request $request){
        try {
            
            $chantier = Chantier::find($request->id);
            if(isset($chantier)){
                
                DB::beginTransaction();
                
                $chantier->delete();
                
                DB::commit();
    
                return redirect()->route("admin.chantiers")->with("success","Success! Chantier Supprimé.");
            
            }else{
                return redirect()->route("admin.chantiers")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }



    public function chantierDetails(Request $request){
        try {
        
            $chantier = Chantier::find($request->id);

            if(isset($chantier)){


                $employees = User::where('role', 'employee')->get();

                $taches = Tache::where('chantier_id',$chantier->id)->get();

                return view('admin.chantier-details', compact('chantier', 'taches', 'employees'));
            }else{
                return redirect()->route("admin.chantier-report")->with("error","Oops! Something Went Wrong. Try Again!");
            }
            
        } catch (\Exception $e) {
            \Log::error('Error occurred: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function insertTacheChantier(Request $request){
        $request->validate([
            'chantier_id' =>'required',
            'nom' =>'required',
            'date_debut' =>'required|date',
            'date_fin' =>'required|date|after_or_equal:date_debut',
            'status' =>'required',
            'photos.*' => 'max:2048',
        ]);
        try {


            $chantier = Chantier::find($request->chantier_id);
    
            if(isset($chantier)){
    
    
                $errors = "";
    
    
                if ($errors) {
                    return redirect()->back()->with("error", $errors);
                }
                else{
                    DB::beginTransaction();
                    
                    $tache = new Tache();
                    $tache->chantier_id = $request->chantier_id;
                    $tache->nom = $request->nom;
                    $tache->description = $request->description;
                    $tache->date_debut = $request->date_debut;
                    $tache->date_fin = $request->date_fin;
                    $tache->nb_jours = $this->calculateWorkingDays($request->date_debut, $request->date_fin);
                    $tache->status = $request->status;
                    
                    if(isset($request->employees)){
                        $tache->employees = json_encode($request->employees);
                    }
    
                    if ($request->hasFile('photos')) {
                        $fileNames = []; // Initialize an array to hold the filenames
                
                        foreach ($request->file('photos') as $file) {
                            $file_name = "TacheChantier-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                            $destinationPath = 'uploads';
                            $file->move($destinationPath, $file_name);
                            $fileNames[] = $file_name; // Add the filename to the array
                        }
                
                        $tache->photo = json_encode($fileNames); // Save filenames as JSON
                    }
    
                    $tache->save();
                    
                    DB::commit();
            
                    return redirect()->back()->with('success','Success! Tache Ajouté.');
                }
            } else{
                return redirect()->route("admin.chantiers")->with("error","Oops! Something Went Wrong. Try Again!");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }

    }

    public function editTacheChantier(Request $request){
        $tache = Tache::find($request->id);
        if(isset($tache)){
            $employees = User::where('role', 'employee')->get();
            return view("admin.edit-tache-chantier", compact("tache","employees"));
        }else{
            return redirect()->back()->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }

    public function updateTacheChantier(Request $request){
        $request->validate([
            'id' => 'required',
            'nom' =>'required',
            'date_debut' =>'required|date',
            'date_fin' =>'required|date|after_or_equal:date_debut',
            'status' =>'required',
            'photos.*' => 'max:2048',
        ]);
        
        try {

            $tache = Tache::find($request->id);
    
            if(isset($tache)){
                DB::beginTransaction();
                
                $tache->nom = $request->nom;
                $tache->description = $request->description;
                $tache->date_debut = $request->date_debut;
                $tache->date_fin = $request->date_fin;
                $tache->nb_jours = $this->calculateWorkingDays($request->date_debut, $request->date_fin);
                $tache->status = $request->status;
                
                if(isset($request->employees)){
                    $tache->employees = json_encode($request->employees);
                }else{
                    $tache->employees = [];
                }
    
                if ($request->hasFile('photos')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photos') as $file) {
                        $file_name = "TacheChantier-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    $tache->photo = json_encode($fileNames); // Save filenames as JSON
                }
            
                $tache->save();
                
                DB::commit();
    
                return redirect()->route("admin.chantier-details", $tache->chantier_id)->with("success",'Success! Tache Modifié.');
            } else{
                return redirect()->back()->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function deleteTacheChantier(Request $request){
        try {
            $tache = Tache::find($request->id);
            if(isset($tache)){
                
                DB::beginTransaction();
                
                $tache->delete();
                
                DB::commit();
    
                return redirect()->back()->with("success","Success! Tache Supprimé.");
            
            }else{
                return redirect()->back()->with("error","Oops! Something Went Wrong. Try Again!");
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }



    public function familles(){
        $familles = Famille::all();
        
        return view('admin.familles',compact('familles'));
    }

    public function detailsFamille(Request $request){
        $famille = Famille::find($request->id);
        if(isset($famille)){
            
            return view('admin.famille-details',compact('famille'));
        
        }else{
            return redirect()->back()->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }

    public function addFamille(Request $request){
        $lieux = Lieu::all();
        $elements = Element::all();

        return view('admin.add-famille',compact('lieux', 'elements'));
    }

    public function insertFamille(Request $request){
        $request->validate([
            'nom' =>'required',
            'lieuId' =>'required',
            'image.*' => 'max:2048',
        ]);
        
        try {
                
    
            $errors = "";
    
    
            if ($errors) {
                return redirect()->back()->with("error", $errors);
            }
            else{
                
                DB::beginTransaction();
                
                $famille = new Famille();
                $famille->nom = $request->nom;
    
                $lieu = Lieu::findOrFail($request->lieuId);
                $famille->lieu = $lieu->id;
    
                if ($request->hasFile('image')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('image') as $file) {
                        $file_name = "Famille-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $famille->image = json_encode($fileNames); // Save filenames as JSON
                }
    
                $famille->elements = "[]";
                $famille->save();
                
                DB::commit();
        
                return redirect()->route('admin.familles')->with('success','Success! Famille Ajouté.');
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }
    public function editFamille(Request $request){
        $famille = Famille::find($request->id);
        $lieux = Lieu::all();

        $elements = Element::all();

        if(isset($famille)){
            return view("admin.edit-famille", compact("famille", "lieux", "elements"));
        }else{
            return redirect()->route("admin.familles")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateFamille(Request $request){
        $request->validate([
            'id' => 'required',
            'nom' =>'required',
            'lieuId' =>'required',
            'image.*' => 'max:2048',
        ]);
        
        try {

            $famille = Famille::find($request->id);
    
            if(isset($famille)){
                
                DB::beginTransaction();
                
                $famille->nom = $request->nom;
                
                
                $lieu = Lieu::findOrFail($request->lieuId);
                $famille->lieu = $lieu->id;
    
                if ($request->hasFile('image')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('image') as $file) {
                        $file_name = "Famille-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $famille->image = json_encode($fileNames); // Save filenames as JSON
                }
            
                $famille->save();
                
                DB::commit();
    
                return redirect()->route("admin.familles")->with("success",'Success! Famille Modifié.');
            } else{
                return redirect()->route("admin.familles")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteFamille(Request $request){
        
        try {
            $famille = Famille::find($request->id);
            
            if(isset($famille)){
                
                DB::beginTransaction();
                
                $famille->delete();
                
                DB::commit();
    
                return redirect()->route("admin.familles")->with("success","Success! Famille Supprimé.");
            
            }else{
                return redirect()->route("admin.familles")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }



    
    public function lieux(){
        $lieux = Lieu::all();
        
        return view('admin.lieux',compact('lieux'));
    }

    public function addLieu(Request $request){
        return view('admin.add-lieu');
    }

    public function insertLieu(Request $request){
        $request->validate([
            'lieu' =>'required',
            'photo.*' => 'max:2048',
        ]);
        try {
            $errors = "";
    
    
            if ($errors) {
                return redirect()->back()->with("error", $errors);
            }
            else{
                DB::beginTransaction();
                
                $lieu = new Lieu();
                $lieu->lieu = $request->lieu;
    
                if ($request->hasFile('photo')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photo') as $file) {
                        $file_name = "Lieu-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $lieu->photo = json_encode($fileNames); // Save filenames as JSON
                }
    
                $lieu->save();
                
                DB::commit();
        
                return redirect()->route('admin.lieux')->with('success','Success! Lieu Ajouté.');
            }

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }
    public function editLieu(Request $request){
        $lieu = Lieu::find($request->id);
        if(isset($lieu)){
            return view("admin.edit-lieu", compact("lieu"));
        }else{
            return redirect()->route("admin.lieux")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateLieu(Request $request){
        $request->validate([
            'id' => 'required',
            'lieu' =>'required',
            'photo.*' => 'max:2048',
        ]);
        
        try {


            $lieu = Lieu::find($request->id);
    
            if(isset($lieu)){
                DB::beginTransaction();
                
                $lieu->lieu = $request->lieu;
    
                if ($request->hasFile('photo')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photo') as $file) {
                        $file_name = "Lieu-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $lieu->photo = json_encode($fileNames); // Save filenames as JSON
                }
            
                $lieu->save();
                DB::commit();
    
                return redirect()->route("admin.lieux")->with("success",'Success! Lieu Modifié.');
            } else{
                return redirect()->route("admin.lieux")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteLieu(Request $request){
        try {
            $lieu = Lieu::find($request->id);
            if(isset($lieu)){
                DB::beginTransaction();
                // Update all familles that have this lieu to set lieu to null
                Famille::where('lieu', $request->id)->update(['lieu' => null]);
                
                $lieu->delete();
                
                DB::commit();
    
                return redirect()->route("admin.lieux")->with("success","Success! Lieu Supprimé.");
            
            }else{
                return redirect()->route("admin.lieux")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    public function elements(){
        $elements = Element::all();
        
        return view('admin.elements',compact('elements'));
    }

    public function addElement(Request $request){
        $familles = Famille::all();
        return view('admin.add-element', compact('familles'));
    }

    public function insertElement(Request $request){
        $request->validate([
            'element' =>'required',
            'familleId' =>'required',
            'quantity' =>'required',
            'prix_achat' =>'required',
            'stock_alert' =>'nullable',
            'photo.*' => 'max:2048',
        ]);
        try {
        
            $errors = "";
    
    
            if ($errors) {
                return redirect()->back()->with("error", $errors);
            }
            else{
                
                DB::beginTransaction();
                
                $element = new Element();
                $element->element = $request->element;
    
                if ($request->hasFile('photo')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photo') as $file) {
                        $file_name = "Element-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $element->photo = json_encode($fileNames); // Save filenames as JSON
                }
    
                $element->quantity = $request->quantity;
                // $element->prix = $request->prix;
                $element->stock_alert = $request->stock_alert;
                $element->prix_achat = $request->prix_achat;
                $element->familleId = $request->familleId;
    
                $element->save();
                
    
                //add element to famille
                $famille = Famille::find($request->familleId);
    
                if (isset($famille)) {
                    // Decode the elements JSON into an array
                    $elements = json_decode($famille->elements, true);
    
                    // Ensure $elements is an array
                    if (!is_array($elements)) {
                        $elements = [];
                    }
    
                    // Convert the new element to a string
                    $newElement = (string) $element->id;
    
                    // Add the new element if it's not already in the array
                    if (!in_array($newElement, $elements)) {
                        $elements[] = $newElement;
                    }
    
                    // Encode the array back to JSON and update the famille
                    $famille->elements = json_encode($elements);
    
                    // Save the updated famille
                    $famille->save();
                }
                
                DB::commit();
    
        
                return redirect()->route('admin.elements')->with('success','Success! Element Ajouté.');
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }
    public function editElement(Request $request){
        $element = Element::find($request->id);
        $familles = Famille::all();
        if(isset($element)){
            return view("admin.edit-element", compact("element","familles"));
        }else{
            return redirect()->route("admin.elements")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }

    public function updateElement(Request $request){
        $request->validate([
            'id' => 'required',
            'element' =>'required',
            'quantity' =>'required',
            'prix_achat' =>'required',
            'stock_alert' =>'nullable',
            'familleId' =>'required',
            'photo.*' => 'max:2048',
        ]);

        try {
            DB::beginTransaction();
            
            $element = Element::find($request->id);
    
            if(isset($element)){
                
                $element->element = $request->element;
    
                if ($request->hasFile('photo')) {
                    $fileNames = []; // Initialize an array to hold the filenames
            
                    foreach ($request->file('photo') as $file) {
                        $file_name = "Element-" . md5(microtime() . uniqid()) . '.' . $file->getClientOriginalExtension();
                        $destinationPath = 'uploads';
                        $file->move($destinationPath, $file_name);
                        $fileNames[] = $file_name; // Add the filename to the array
                    }
            
                    // Assuming you have a `photos` field in your database
                    $element->photo = json_encode($fileNames); // Save filenames as JSON
                }
    
                if(isset($request->familleId)){
    
                    //delete element from old famille
                    $old_famille = Famille::find($element->familleId);
                    // Decode the elements JSON into an array
                    $old_elements = json_decode($old_famille->elements, true);
                    // Ensure $elements is an array
                    if (!is_array($old_elements)) {
                        $old_elements = [];
                    }
    
                    // Convert the new element to a string
                    $oldElementId = (string) $element->id;
    
                    // Remove the element ID from the array if it exists
                    $old_elements = array_filter($old_elements, function ($value) use ($oldElementId) {
                        return $value !== $oldElementId;
                    });
    
                    // Reindex the array to ensure proper JSON encoding
                    $old_elements = array_values($old_elements);
    
                    // Encode the updated array back to JSON and update the old famille
                    $old_famille->elements = json_encode($old_elements);
    
                    // Save the updated famille
                    $old_famille->save();
    
    
                    //add element to famille
                    $famille = Famille::find($request->familleId);
    
                    // Decode the elements JSON into an array
                    $elements = json_decode($famille->elements, true);
    
                    // Ensure $elements is an array
                    if (!is_array($elements)) {
                        $elements = [];
                    }
    
                    // Convert the new element to a string
                    $newElement = (string) $element->id;
    
                    // Add the new element if it's not already in the array
                    if (!in_array($newElement, $elements)) {
                        $elements[] = $newElement;
                    }
    
                    // Encode the array back to JSON and update the famille
                    $famille->elements = json_encode($elements);
    
                    // Save the updated famille
                    $famille->save();
                
                }
                
                
                $element->quantity = $request->quantity;
                // $element->prix = $request->prix;
                $element->stock_alert = $request->stock_alert;
                $element->prix_achat = $request->prix_achat;
                $element->familleId = $request->familleId;
    
                $element->save();
                DB::commit();
    
                return redirect()->route("admin.elements")->with("success",'Success! Element Modifié.');
            } else{
                return redirect()->route("admin.elements")->with("error","Oops! Something Went Wrong. Try Again!");
            }
          
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteElement(Request $request){
        try {

            $element = Element::find($request->id);
            if(isset($element)){
                
            DB::beginTransaction();
                // First update all familles that contain this element
                $familles = Famille::whereRaw('JSON_CONTAINS(elements, ?)', ['"'.$request->id.'"'])->get();
                
                foreach($familles as $famille) {
                    $elements = json_decode($famille->elements, true);
                    $elements = array_filter($elements, function($elementId) use ($request) {
                        return $elementId != $request->id;
                    });
                    $famille->elements = json_encode(array_values($elements));
                    $famille->save();
                }
                
                $element->delete();
                DB::commit();
    
                return redirect()->route("admin.elements")->with("success","Success! Element Supprimé.");
            
            }else{
                return redirect()->route("admin.elements")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    

    public function materiels(){
        $materiels = Materiel::all();
        
        return view('admin.materiels',compact('materiels'));
    }

    public function addMateriel(){
        return view('admin.add-materiel');
    }

    public function insertMateriel(Request $request){
        $request->validate([
            'materiel' =>'required',
            'reference' =>'required',
            'photo.*' => 'max:2048',
        ]);
        
        try {
            DB::beginTransaction();


            $materiel = new Materiel();
            $materiel->reference = $request->reference;
            $materiel->materiel = $request->materiel;
            $materiel->prix_achat = $request->prix_achat;
            $materiel->status = 'disponible'; // Set default status to disponible
    
            if($request->hasFile('photo')){
                $files = $request->file('photo');
                $fileNames = [];
                foreach($files as $file){
                    $fileName = time().'_'.$file->getClientOriginalName();
                    $file->move(public_path('uploads'), $fileName);
                    $fileNames[] = $fileName;
                }
                $materiel->photo = json_encode($fileNames);
            }
    
            $materiel->save();
            
            DB::commit();
    
            return redirect()->route('admin.materiels')->with('success','Success! Materiel Ajouté.');
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function editMateriel(Request $request){
        $materiel = Materiel::find($request->id);
        if(isset($materiel)){
            return view("admin.edit-materiel", compact("materiel"));
        }else{
            return redirect()->route("admin.materiels")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }

    public function updateMateriel(Request $request){
        $request->validate([
            'id' => 'required',
            'materiel' =>'required',
            'reference' =>'required',
            'photo.*' => 'max:2048',
        ]);

        try {

            $materiel = Materiel::find($request->id);
    
            if(isset($materiel)){
                DB::beginTransaction();
                
                $materiel->reference = $request->reference;
                $materiel->materiel = $request->materiel;
                $materiel->prix_achat = $request->prix_achat;
    
                if($request->hasFile('photo')){
                    $files = $request->file('photo');
                    $fileNames = [];
                    foreach($files as $file){
                        $fileName = time().'_'.$file->getClientOriginalName();
                        $file->move(public_path('uploads'), $fileName);
                        $fileNames[] = $fileName;
                    }
                    $materiel->photo = json_encode($fileNames);
                }
    
                $materiel->save();
                
                DB::commit();
    
                return redirect()->route('admin.materiels')->with('success','Success! Materiel Modifié.');
            }else{
                return redirect()->route('admin.materiels')->with('error','Oops! Something Went Wrong. Try Again!');
            }
            
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    
    public function deleteMateriel(Request $request){
        try {
            
            $materiel = Materiel::find($request->id);
            if(isset($materiel)){
                
                DB::beginTransaction();
                $materiel->delete();
                DB::commit();
    
                return redirect()->route("admin.materiels")->with("success","Success! Matériel Supprimé.");
            
            }else{
                return redirect()->route("admin.materiels")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    public function addMaterielSalarie(){
        $salaries = User::where('role', 'employee')->get();
        $materiels = Materiel::all();
        return view('admin.affectation-materiel-salarie',compact('salaries','materiels'));
    }

    public function insertMaterielSalarie(Request $request){
        $request->validate([
            'salarieId' =>'required',
            'materielId' =>'required',
            'date_de_prise' =>'required',
            'date_de_remise' =>'required',
        ]);

        try {
            $materiel = Materiel::find($request->materielId);
            
            if($materiel->status=="disponible"){
                DB::beginTransaction();

                $affectation = new MaterielSalarie();
                $affectation->salarie_id = $request->salarieId;
                $affectation->materiel_id = $request->materielId;
                $affectation->date_de_prise = $request->date_de_prise;
                $affectation->date_de_remise = $request->date_de_remise;
                $affectation->status = "assigned";
                $affectation->save();

                // Update materiel status
                $materiel->status = "indisponible";
                $materiel->save();
                
                DB::commit();

                return redirect()->route('admin.affectations')->with('success','Success! Affectation Ajouté.');
            } else {
                return redirect()->route('admin.affectations')->with('error',"Ce matériel n'est pas disponible!");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.affectations')->with('error', "Une erreur s'est produite lors de l'affectation.");
        }
    }

    public function affectations()
    {
        $affectations = MaterielSalarie::orderBy('created_at', 'desc')->get();
        return view('admin.affectation', compact('affectations'));
    }

    public function returnMateriel($id)
    {
        try {
            $affectation = MaterielSalarie::find($id);
            if($affectation) {
                DB::beginTransaction();
                // Update the affectation status and return date
                $affectation->status = 'returned';
                $affectation->date_retour = now();
                $affectation->save();
    
                // Update the materiel status to disponible
                $materiel = Materiel::find($affectation->materiel_id);
                if($materiel) {
                    $materiel->status = 'disponible';
                    $materiel->save();
                }
                DB::commit();
    
                return redirect()->route('admin.affectations')->with('success', 'Matériel marqué comme retourné avec succès.');
            }
            return redirect()->route('admin.affectations')->with('error', 'Affectation non trouvée.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.affectations')->with('error', "Une erreur s'est produite lors de l'affectation.");
        }
    }

    public function deleteAffectation($id)
    {
        try {
            $affectation = MaterielSalarie::find($id);
            if($affectation) {
                // Update the materiel status to disponible
                $materiel = Materiel::find($affectation->materiel_id);
                
                DB::beginTransaction();
                if($materiel) {
                    $materiel->status = 'disponible';
                    $materiel->save();
                }
                $affectation->delete();
                
                DB::commit();
                return redirect()->route('admin.affectations')->with('success', 'Affectation supprimée avec succès.');
            }
            return redirect()->route('admin.affectations')->with('error', 'Affectation non trouvée.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    public function calculateEmployeeStats($start_date, $end_date, $chantier_id = null)
    {
        $start_date = Carbon::parse($start_date)->startOfDay();
        $end_date = Carbon::parse($end_date)->endOfDay();
        
        $employees = User::where('role', 'employee')->get();
        $stats = [];
        
        foreach ($employees as $employee) {
            $employeeStats = [
                'id' => $employee->id,
                'name' => $employee->name . ' ' . $employee->prenom,
                'total_work_days' => 0,
                'total_absence_days' => 0,
                'real_worked_days' => 0
            ];
            
            // Calculate total work days from EVENTS (not tasks)
            $eventsQuery = Event::where('employee_id', $employee->id)
                ->where(function($query) use ($start_date, $end_date) {
                    $query->where(function($q) use ($start_date, $end_date) {
                        $q->whereBetween('date_debut', [$start_date, $end_date])
                        ->orWhereBetween('date_fin', [$start_date, $end_date])
                        ->orWhere(function($q) use ($start_date, $end_date) {
                            $q->where('date_debut', '<=', $start_date)
                                ->where('date_fin', '>=', $end_date);
                        });
                    });
                });
                
            // Apply chantier filter if specified
            if ($chantier_id) {
                $eventsQuery->where('chantier_id', $chantier_id);
            }
            
            $events = $eventsQuery->get();

            foreach ($events as $event) {
                $event_start = Carbon::parse($event->date_debut);
                $event_end = Carbon::parse($event->date_fin);
                
                $period_start = $event_start->max($start_date);
                $period_end = $event_end->min($end_date);
                
                if ($period_start <= $period_end) {
                    $workDays = $this->calculateWorkingDays($period_start, $period_end);
                    $employeeStats['total_work_days'] += $workDays;
                }
            }
            
            // Calculate total absence days from ABSENCES
            $absencesQuery = Absence::where('employee_id', $employee->id)
                ->where(function($query) use ($start_date, $end_date) {
                    $query->where(function($q) use ($start_date, $end_date) {
                        $q->whereBetween('date_debut', [$start_date, $end_date])
                        ->orWhereBetween('date_fin', [$start_date, $end_date])
                        ->orWhere(function($q) use ($start_date, $end_date) {
                            $q->where('date_debut', '<=', $start_date)
                                ->where('date_fin', '>=', $end_date);
                        });
                    });
                });
                
            // Apply chantier filter if specified
            if ($chantier_id) {
                $absencesQuery->where('chantier_id', $chantier_id);
            }
            
            $absences = $absencesQuery->get();

            foreach ($absences as $absence) {
                $absence_start = Carbon::parse($absence->date_debut);
                $absence_end = Carbon::parse($absence->date_fin);
                
                $period_start = $absence_start->max($start_date);
                $period_end = $absence_end->min($end_date);
                
                if ($period_start <= $period_end) {
                    $absenceDays = $this->calculateWorkingDays($period_start, $period_end);
                    $employeeStats['total_absence_days'] += $absenceDays;
                }
            }
            
            $employeeStats['real_worked_days'] = max(0, $employeeStats['total_work_days'] - $employeeStats['total_absence_days']);
            
            // Only include employees with any activity in this chantier
            if (!$chantier_id || $employeeStats['total_work_days'] > 0) {
                $stats[] = $employeeStats;
            }
        }
        
        return $stats;
    }

        public function comptabilite(Request $request)
    {
        try {
            $start_date = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
            $end_date = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
            $chantier_id = $request->input('chantier_id');
            
            $employeeStats = $this->calculateEmployeeStats($start_date, $end_date, $chantier_id);
            
            // Get all chantiers for the dropdown
            $chantiers = Chantier::all();
            
            // Get the selected chantier name if applicable
            $selectedChantier = null;
            if ($chantier_id) {
                $selectedChantier = Chantier::find($chantier_id);
            }
            
            return view('admin.comptabilite', compact(
                'employeeStats', 
                'start_date', 
                'end_date', 
                'chantiers', 
                'chantier_id', 
                'selectedChantier'
            ));
            
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function absences(Request $request)
    {
        $query = Absence::with(['employee']);
        
        // Default to current month if no dates provided
        $start_date = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $end_date = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        
        // Apply date range filter
        $query->where(function($q) use ($start_date, $end_date) {
            $q->whereBetween('date_debut', [$start_date, $end_date])
            ->orWhereBetween('date_fin', [$start_date, $end_date])
            ->orWhere(function($q) use ($start_date, $end_date) {
                $q->where('date_debut', '<=', $start_date)
                    ->where('date_fin', '>=', $end_date);
            });
        });
        
        $absences = $query->orderBy('date_debut', 'desc')->get();
        $employees = User::where('role', 'employee')->get();
        $chantiers = Chantier::all();
        
        return view('admin.absences', compact('absences', 'employees', 'chantiers', 'start_date', 'end_date'));
    }

    public function insertAbsence(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'type' => 'required|in:absence,conge_individuel,conge_collectif',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'reason' => 'nullable|string',
            'statut' => 'required|in:pending,approved,rejected'
        ]);
        
        try {
            $data = $request->all();
            $data['jours_absence'] = $this->calculateWorkingDays($request->date_debut, $request->date_fin);
            
            DB::beginTransaction();
            Absence::create($data);
            DB::commit();
            
            return redirect()->route('admin.absences')->with('success', 'Absence ajoutée avec succès');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function editAbsence($id)
    {
        try {
            $absence = Absence::findOrFail($id);
        
            if(isset($absence)){
                
                $employees = User::where('role', 'employee')->get();
                return view('admin.edit-absence', compact('absence', 'employees'));
            
            }else{
                return redirect()->back()->with('error', 'Une Erreur est survenue!!!');
            }
        } catch (\Exception $e) {
            
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function updateAbsence(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'employee_id' => 'required|exists:users,id',
            'type' => 'required|in:absence,conge_individuel,conge_collectif',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'statut' => 'required|in:pending,approved,rejected',
            'reason' => 'nullable|string'
        ], [
            'date_fin.after_or_equal' => 'La date de fin doit être égale ou postérieure à la date de début'
        ]);

        try {
            $absence = Absence::findOrFail($request->absence_id);
            $data = $request->except('absence_id');
            
            $data['jours_absence'] = $this->calculateWorkingDays($request->date_debut, $request->date_fin);

            DB::beginTransaction();
            $absence->update($data);
            DB::commit();
            
            return redirect()->route('admin.absences')->with('success', 'Absence mise à jour avec succès');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function deleteAbsence($id)
    {
        try {
            $absence = Absence::findOrFail($id);
            if(isset($absence)){
                
                DB::beginTransaction();
                $absence->delete();
                DB::commit();
                
                return redirect()->route('admin.absences')->with('success', 'Absence supprimée avec succès');
                
            }else{
                return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function getChantierTaches($id)
    {
        $taches = Tache::where('chantier_id', $id)->get();
        return response()->json($taches);
    }

    // Planning-related methods
    public function planning()
    {
        $metiers = Metier::all();
        $employees = User::where('role', 'employee')->get();
        $chantiers = Chantier::with(['taches'])->get();
        
        $events = $chantiers->map(function($chantier) {

            // Add one day to the end date to make it inclusive
            $endDate = new DateTime($chantier->date_de_fin);
            $endDate->modify('+1 day');
            $endDate = $endDate->format('Y-m-d');
            
            return [
                'id' => $chantier->id,
                'title' => $chantier->nom,
                'start' => $chantier->date_de_debut,
                'end' => $endDate,
                'backgroundColor' => $this->getStatusColor($chantier->status),
                'allDay' => true,
                'extendedProps' => [
                    'adresse' => $chantier->adresse,
                    'nbjours' => $chantier->nbjours,
                    'taches' => $chantier->taches ? $chantier->taches->toArray() : []
                ]
            ];
        });

        return view('admin.planning', compact('events', 'employees', 'metiers'));
    }

    public function getChantierDetails($id)
    {
        $chantier = Chantier::with(['taches' => function($query) {
            $query->orderBy('date_debut', 'asc');
        }])->findOrFail($id);
        
        // Get all employees
        $allEmployees = User::where('role', 'employee')->get();
        
        // Add all employees to the response
        $chantier->all_employees = $allEmployees;
        
        return response()->json($chantier);
    }

    public function getSalarieDetails($id)
    {
        try {
            $parts = explode('-', $id);
            if (count($parts) !== 3) {
                throw new \Exception('Invalid ID format');
            }

            $employeeId = $parts[0];
            $year = $parts[1];
            $month = $parts[2];

            $employee = User::findOrFail($employeeId);
            $startDate = Carbon::create($year, $month, 1)->startOfMonth();
            $endDate = $startDate->copy()->endOfMonth();

            // Get all tasks for this employee in the specified month
            $taches = Tache::whereRaw('JSON_CONTAINS(employees, ?)', ['"' . $employeeId . '"'])
                ->whereHas('chantier', function($query) use ($startDate, $endDate) {
                    $query->where(function($q) use ($startDate, $endDate) {
                        $q->whereBetween('date_de_debut', [$startDate, $endDate])
                          ->orWhereBetween('date_de_fin', [$startDate, $endDate])
                          ->orWhere(function($q) use ($startDate, $endDate) {
                              $q->where('date_de_debut', '<=', $startDate)
                                ->where('date_de_fin', '>=', $endDate);
                          });
                    });
                })
                ->with(['chantier' => function($query) {
                    $query->select('id', 'nom', 'date_de_debut', 'date_de_fin');
                }])
                ->get();

            return response()->json([
                'id' => $employee->id,
                'name' => $employee->prenom . ' ' . $employee->name,
                'taches' => $taches->map(function($tache) use ($startDate, $endDate) {
                    $taskStart = Carbon::parse($tache->date_debut);
                    $taskEnd = Carbon::parse($tache->date_fin);
                    
                    // Calculate working days for this task in the current month
                    $start = max($taskStart, $startDate);
                    $end = min($taskEnd, $endDate);
                    
                    return [
                        'id' => $tache->id,
                        'nom' => $tache->nom,
                        'date_debut' => $start->format('Y-m-d'),
                        'date_fin' => $end->format('Y-m-d'),
                        'nb_jours' => $this->calculateWorkingDays($start->format('Y-m-d'), $end->format('Y-m-d')),
                        'chantier' => $tache->chantier
                    ];
                })
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function calculateWorkingDays($startDate, $endDate)
    {
        $start = new DateTime($startDate);
        $end = new DateTime($endDate);
        $interval = new DateInterval('P1D');
        $dateRange = new DatePeriod($start, $interval, $end->modify('+1 day'));

        $workingDays = 0;
        foreach ($dateRange as $date) {
            // Skip weekends (6 = Saturday, 0 = Sunday)
            if ($date->format('N') < 6) {
                $workingDays++;
            }
        }
        return $workingDays;
    }

    public function planningSalarie(Request $request)
    {
        // try {
            $employees = User::where('role', 'employee')->get();
            $selectedEmployee = $request->input('employee_id');
    
            $query = Tache::with('chantier')
                ->whereNotNull('date_debut')
                ->whereNotNull('date_fin');
    
            if ($selectedEmployee) {
                $query->whereRaw('JSON_CONTAINS(employees, ?)', ['"' . $selectedEmployee . '"']);
            }
    
            $tasks = $query->get();
    
            $events = [];
            foreach ($tasks as $task) {
                $endDate = new DateTime($task->date_fin);
                $endDate->modify('+1 day');
                $endDate = $endDate->format('Y-m-d');

                // Get employee names for this task
                $employeeIds = $task->employees ?? []; // Already decoded by accessor
                $taskEmployees = User::whereIn('id', $employeeIds)->get();
                $employeeNames = $taskEmployees->pluck('name')->implode(', ');
    
                $events[] = [
                    'id' => $task->id,
                    'title' => $task->nom . ' (' . $task->chantier->nom . ')',
                    'start' => $task->date_debut,
                    'end' => $endDate,
                    'description' => "Employés: " . $employeeNames,
                    'backgroundColor' => $task->chantier->color ?? '#3788d8',
                    'borderColor' => $task->chantier->color ?? '#3788d8',
                    'textColor' => '#ffffff',
                    'url' => route('admin.edit-tache-chantier', ['id' => $task->id]),
                    'extendedProps' => [
                        'employee_name' => $employeeNames,
                        'tache_nom' => $task->nom,
                        'chantier_nom' => $task->chantier->nom,
                        'nb_jours' => $task->nb_jours
                    ]
                ];
            }
    
            // Get absences
            $absenceQuery = Absence::with('employee');
            if ($selectedEmployee) {
                $absenceQuery->where('employee_id', $selectedEmployee);
            }
            $absences = $absenceQuery->get();
    
            foreach ($absences as $absence) {
                $events[] = [
                    'id' => 'absence_' . ($absence->id ?? ''),
                    'title' => 'Absence: ' . ($absence->employee ? $absence->employee->name : 'Employé inconnu'),
                    'start' => $absence->date_debut ?? '',
                    'end' => $absence->date_fin ?? '',
                    'backgroundColor' => '#dc3545',
                    'borderColor' => '#dc3545',
                    'textColor' => '#ffffff',
                    'classNames' => ['absence-event']
                ];
            }
    
            return view('admin.planning-salarie', compact('events', 'employees', 'selectedEmployee'));
            
        
        // } catch (\Exception $e) {
            
        //     return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        // }
    }

    public function updateChantierDates(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:chantiers,id',
            'start' => 'required|date',
            'end' => 'required|date'
        ]);

        $chantier = Chantier::findOrFail($request->id);
        $chantier->date_de_debut = $request->start;
        $chantier->date_de_fin = $request->end;
        
        // Recalculate number of days
        $chantier->nbjours = $this->calculateWorkingDays($request->start, $request->end);
        
        $chantier->save();

        return response()->json(['success' => true]);
    }

    public function storeChantierFromCalendar(Request $request)
    {
        $request->validate([
            'nom' => 'required|string',
            'adresse' => 'required|string',
            'date_de_debut' => 'required|date',
            'date_de_fin' => 'required|date|after_or_equal:date_de_debut'
        ]);

        // Calculate number of working days
        $nbjours = $this->calculateWorkingDays($request->date_de_debut, $request->date_de_fin);

        $chantier = new Chantier();
        $chantier->nom = $request->nom;
        $chantier->adresse = $request->adresse;
        $chantier->date_de_debut = $request->date_de_debut;
        $chantier->date_de_fin = $request->date_de_fin;
        $chantier->nbjours = $nbjours;
        $chantier->save();

        return response()->json([
            'success' => true,
            'chantier' => $chantier
        ]);
    }

    private function getStatusColor($status)
    {
        switch ($status) {
            case 'in_progress':
                return '#3498db'; // Blue
            case 'completed':
                return '#2ecc71'; // Green
            case 'pending':
                return '#ce932a'; // Yellow
            default:
                return '#95a5a6'; // Grey
        }
    }

    /////////////


    public function admins(){
        $admins = User::where('role', 'admin')->get();
        
        return view('admin.admins',compact('admins'));
    }

    public function addAdmin(Request $request){
        return view('admin.add-admin');
    }

    public function insertAdmin(Request $request){
        $request->validate([
            'nom' => 'required',
            'prenom' => 'required',
            'email' =>'required | unique:users,email',
            'password' => 'required',
        ]);

        try {
            DB::beginTransaction();
            $admin = new User();
            $admin->name = $request->nom;
            $admin->prenom = $request->prenom;
            $admin->email = $request->email;
            $admin->password = $request->password;
            $admin->role = 'admin';
            $admin->save();
            DB::commit();


            return redirect()->route('admin.admins')->with('success','Admin Ajouté avec succès !');
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    
    public function editAdmin(Request $request){
        $admin = User::find($request->id);
        if(isset($admin)){
            return view("admin.edit-admin",compact("admin"));
        }else{
            return redirect()->route("admin.admins")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateAdmin(Request $request){
        $request->validate([
            'id' => 'required',
            'name' => 'required',
            'prenom' => 'required',
            'email' =>'required',
            'password' => 'required',
        ]);
        try {
            $admin = User::find($request->id);
            $oldemail = $admin->email;
            $oldpassword = $admin->password;
            if(isset($admin)){
                
                DB::beginTransaction();
                $admin->name = $request->name;
                $admin->prenom = $request->prenom;
                if(isset($request->email) && $request->email != $oldemail){
                    $admin->email = $request->email;
                }
                
                if(isset($request->password) && $request->password != $oldpassword){
                    $admin->password = $request->password;
                }
                
                $admin->role = 'admin';
                $admin->update();
                DB::commit();
                
                return redirect()->route('admin.admins')->with("success","Success! Admin Updated Successfuly!");
            }else{
                return redirect()->route("admin.admins")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    public function deleteAdmin(Request $request){
        try {
            $admin = User::find($request->id);
            if(isset($admin)){
                
                DB::beginTransaction();
                $admin->delete();
                DB::commit();
                
                return redirect()->route('admin.admins')->with("success","Success! Admin Deleted Successfuly!");
            } else{
                return redirect()->route("admin.admins")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    
    public function manageEmployees(Request $request){
        $employees = User::where('role','employee')->orderBy('id','desc')->get();
        return view('admin.manage-employees',compact('employees'));
    }

    public function addEmployee(Request $request){
        $metiers = Metier::all();
        return view("admin.add-employee",compact("metiers"));
    }
    public function insertEmployee(Request $request){
        $request->validate([
            'nom' =>'required',
            'prenom' =>'required',
            'email' =>'required | unique:users,email',
            'adresse' =>'nullable | string',
            'password' =>'required',
            'phone' =>'required',
            // 'status' =>'required',
            'image' =>'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            
        ]);
        
        try {

            $text = "";
            $errors = "";
    
            if ($errors) {
                return redirect()->back()->with("error", $errors);
            }
            else{
                DB::beginTransaction();
                
                $employee = new User();
                $employee->name = $request->nom;
                $employee->prenom = $request->prenom;
                $employee->phone = $request->phone;
                $employee->email = $request->email;
                $employee->phone = $request->phone;
                $employee->adresse = $request->adresse;
                $employee->password = $request->password;
                if(isset($request->is_couvreur)){
                    $employee->is_couvreur = 1;
                }else{
                    $employee->is_couvreur = 0;
                }
                $employee->status = 'Inactif';//$request->status
    
                $employee->role = 'employee';
    
                if(isset($request->primary_metiers)){
                    $employee->primary_metiers = json_encode($request->primary_metiers);
                }else{
                    $employee->primary_metiers = "[]";
                }
    
                if(isset($request->secondary_metiers)){
                    $employee->secondary_metiers = json_encode($request->secondary_metiers);
                }else{
                    $employee->secondary_metiers = "[]";
                }
                
                $text .= "Employee Name: ".$employee->name. " Phone: ".$employee->phone. " Email: ".$employee->email;
                if(isset($request->password)){
                    $employee->password = $request->password;
                    $text .= " Password: ". $employee->password;
                }
    
                if($request->hasFile('image')){
                    $file = $request->file('image');
                    $file_name = $file->getClientOriginalName();
                    $destinationPath = public_path('media');
    
                    $file->move($destinationPath,$file_name);
    
                    $employee->image = $file_name;
                }
    
    
                $employee->save();
    
                // $user = User::find(session('current_admin')->id);
                // $log = new Log();
                // $log->details = "Admin: ".$user->name." Added employee: ".$text;
                // $log->user_id = $user->id;
                // $log->save();
                
                
                DB::commit();
    
                return redirect()->route('admin.manage-employees')->with('success','Success! Employee Ajouté avec succès.');
            }


        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }
    public function editEmployee(Request $request){
        $current_employee = User::find($request->id);
        if(isset($current_employee)){
            //$agents = User::where('role','employee')->get();

            $metiers = Metier::all();
            return view("admin.edit-employee",compact("current_employee","metiers"));
        }else{
            return redirect()->route("admin.manage-employees")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    public function updateEmployee(Request $request){
        $request->validate([
            'employee' => 'required',
            'name' =>'required',
            'prenom' =>'required',
            'email' =>'required | unique:users,email,'.$request->employee,
            'password' =>'required',
            'phone' =>'required',
            'image' =>'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);
        
        // try {
            
            // $text = "";
            $current_employee = User::find($request->employee);
            if (!$current_employee) {
                return redirect()->back()->withErrors('Employee not found');
            }
            if(isset($current_employee)){
                DB::beginTransaction();
                
                $current_employee->name = $request->name;
                $current_employee->prenom = $request->prenom;
                $current_employee->adresse = $request->adresse;
                $current_employee->phone = $request->phone;
                $current_employee->email = $request->email;
                // $current_employee->password = 'RenovImmo2025...';
                if(isset($request->is_couvreur)){
                    $current_employee->is_couvreur = 1;
                }else{
                    $current_employee->is_couvreur = 0;
                }
                $current_employee->status = 'Inactif';//$request->status
                
                // $text .= "Employee Name: ".$current_employee->name. " Phone: ".$current_employee->phone. " Email: ".$current_employee->email;
                if(isset($request->password)){
                    $current_employee->password = $request->password;
                    // $text .= " Mot de passe: ". $current_employee->password;
                }
    
                if(isset($request->primary_metiers)){
                    $current_employee->primary_metiers = json_encode($request->primary_metiers);
                }else{
                    $current_employee->primary_metiers = "[]";
                }
    
                if(isset($request->secondary_metiers)){
                    $current_employee->secondary_metiers = json_encode($request->secondary_metiers);
                }else{
                    $current_employee->secondary_metiers = "[]";
                }
                
                if($request->hasFile('image')){
                    $file = $request->file('image');
                    $file_name = $file->getClientOriginalName();
                    $destinationPath = public_path('media');
    
                    $file->move($destinationPath,$file_name);
    
                    $current_employee->image = $file_name;
                }
    
                $current_employee->update();
    
                // $user = User::find(session('current_admin')->id);
                // $log = new Log();
                // $log->details = "Admin: ".$user->name." Edited employee: ".$text;
                // $log->user_id = $user->id;
                // $log->save();
                DB::commit();
    
                return redirect()->route("admin.manage-employees")->with("success",'Success! Employee Updated Successfully.');
            }else{
                return redirect()->route("admin.manage-employees")->with("error","Oops! Something Went Wrong. Try Again!");
            }
            
        // } catch (\Exception $e) {
        //     DB::rollBack();

        //     return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        // }
    }
    
    
    public function deleteEmployee(Request $request){
        try {
            $current_employee = User::find($request->id);
            if(isset($current_employee)){
                
                DB::beginTransaction();
                $current_employee->delete();
    
                $user = User::find(session('current_admin')->id);
                $log = new Log();
                $log->details = "Admin: ".$user->name." Deleted Employee: ".$current_employee->name;
                $log->user_id = $user->id;
                $log->save();
                DB::commit();
    
                return redirect()->route("admin.manage-employees")->with("success","Success! Employee Deleted Successfully.");
            
            }else{
                return redirect()->route("admin.manage-employees")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function stockMovement($id)
    {
        $element = Element::findOrFail($id);
        if(isset($element)){
                    
            $movements = StockMovement::where('element_id', $id)
                ->orderBy('created_at', 'desc')
                ->get();
    
            return view('admin.stock-movement', compact('element', 'movements'));
        }
        else{
            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }

    public function addStock(Request $request)
    {
        $request->validate([
            'element_id' => 'required|exists:elements,id',
            'quantity' => 'required|integer|min:1',
            'reference' => 'nullable|string',
            'notes' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            // Create stock movement record
            StockMovement::create([
                'element_id' => $request->element_id,
                'type' => 'in',
                'quantity' => $request->quantity,
                'reference' => $request->reference,
                'notes' => $request->notes,
                'user_id' => session('current_admin')->id
            ]);

            // Update element quantity
            $element = Element::find($request->element_id);
            $element->quantity += $request->quantity;
            $element->save();

            DB::commit();
            return redirect()->back()->with('success', 'Stock ajouté avec succès');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()->with('error', 'Erreur lors de l\'ajout du stock');
        }
    }

    public function removeStock(Request $request)
    {
        $request->validate([
            'element_id' => 'required|exists:elements,id',
            'quantity' => 'required|integer|min:1',
            'reference' => 'nullable|string',
            'notes' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $element = Element::find($request->element_id);
            
            // Check if we have enough stock
            if ($element->quantity < $request->quantity) {
                return redirect()->back()->with('error', 'Stock insuffisant');
            }

            // Create stock movement record
            StockMovement::create([
                'element_id' => $request->element_id,
                'type' => 'out',
                'quantity' => $request->quantity,
                'reference' => $request->reference,
                'notes' => $request->notes,
                'user_id' => session('current_admin')->id
            ]);

            // Update element quantity
            $element->quantity -= $request->quantity;
            $element->save();

            DB::commit();
            return redirect()->back()->with('success', 'Stock retiré avec succès');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()->with('error', 'Erreur lors du retrait du stock');
        }
    }

    /////////////product


    public function addProduct(Request $request){
        return view('admin.add-product');
    }
    public function insertProduct(Request $request){
        $request->validate([
            //'date' => 'required',
            'name' => 'required',
            'description' => 'required',
            'price' => 'required',
            'sell_price' => 'required',
            'status' => 'required',
            'media' => 'file|mimes:jpg,jpeg,png|max:4200', // 50MB max
        ]);

        try {
            DB::beginTransaction();
            
            $product = new Product();
            $product->product_id = uniqid('PRD-');
            $product->name = $request->name;
            $product->description = $request->description;
            // $product->date = Date('Y-m-d');
            $product->price = $request->price;
            $product->sell_price = $request->sell_price;
            $product->status = $request->status;
    
    
            if(isset($request->picture)){
                $file = $request->file('media');
                $file_name = "Product-".md5(microtime()).'.'.$file->getClientOriginalExtension();
                $destinationpath = 'layout/assets/products';
                $file->move($destinationpath,$file_name);
                $product->media = $file_name;
            }
            
            $product->save();
            DB::commit();
            
            return redirect()->route('admin.product-report')->with('success','Product added successfully !');
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }

    }

    public function editProduct(Request $request){
        $product = Product::find($request->id);
        if(isset($product)){
            return view('admin.edit-product',compact('product'));
        }else{
            return redirect()->route("admin.product-report")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }

    public function updateProduct(Request $request){
        $request->validate([
            //'date' => 'required',
            'name' => 'required',
            'description' => 'required',
            'price' => 'required',
            'sell_price' => 'required',
            'status' => 'required',
            'media' => 'file|mimes:jpg,jpeg,png|max:4200', 
            
        ]);
        try {
        
            $product = Product::find($request->id);
            if(isset($product)){
                DB::beginTransaction();
                
                $product->name = $request->name;
                $product->description = $request->description;
                $product->price = $request->price;
                $product->sell_price = $request->sell_price;
                $product->status = $request->status;
    
                if(isset($request->media)){
                    $file = $request->file('media');
                    $file_name = "Product-".md5(microtime()).'.'.$file->getClientOriginalExtension();
                    $destinationpath = 'layout/assets/products';
                    $file->move($destinationpath,$file_name);
                    $product->media = $file_name;
                }
                
                $product->update();
                DB::commit();
    
                return redirect()->route('admin.product-report')->with('success','Success! product Updated Successfully.');
                
            }else{
                return redirect()->route("admin.product-report")->with("error","Oops! Something Went Wrong. Try Again!");
            }
        
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }
    
    
    public function deleteProduct(Request $request){
        try {
            $product = Product::find($request->id);
            
            if(isset($product)){
                DB::beginTransaction();
                
                $product->delete();
                
                $user = User::find(session('current_admin')->id);
                $log = new Log();
                $log->details = "Admin: ".$user->name." Deleted Product ID :".$product->product_id." - Product Name: ".$product->name;
                $log->user_id = $user->id;
                $log->save();
                
                DB::commit();
                
                return redirect()->route('admin.product-report')->with('success','Success! product deleted Successfully.');
            }else{
                return redirect()->route("admin.product-report")->with("error","Oops! Something Went Wrong. Try Again!");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    public function productDetails(Request $request){
        $product = Product::find($request->id);
        if(isset($product)){
            return view('admin.product-details',compact('product'));
        }else{
            return redirect()->route("admin.product-report")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }


    public function productReport(Request $request){
        $products = Product::orderBy('id','desc')->get();
        return view('admin.product-report',compact('products'));
    }
    

    public function notifications()
    {
        $notifications = Notification::all(); // Fetch all notifications
        return view('admin.notifications', compact('notifications'));
    }

    public function addNotification()
    {
        $employees = User::where('role', 'employee')->get(); // Assuming employees have a 'role' column
        return view('admin.add-notification', compact('employees'));
    }

    public function insertNotification(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|in:info,danger,warning',
            'end_date' => 'nullable|date',
            'content' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'for_all_employees' => 'nullable|boolean',
            'employees' => 'nullable|array',
        ]);
        
        try {
            
            DB::beginTransaction();
            
            $file_name = null;
            if(isset($request->image)){
                $file = $request->file('image');
                $file_name = "Notification-".md5(microtime()).'.'.$file->getClientOriginalExtension();
                $destinationpath = 'public/layout/assets/notifications';
                $file->move($destinationpath,$file_name);
            }
    
            // Create notification
            $notification = Notification::create([
                'title' => $request->title,
                'content' => $request->content,
                'image' => $file_name,
                'type' => $request->type,
                'end_date' => $request->end_date,
                'for_all_employees' => $request->has('for_all_employees'),
            ]);
    
            // Attach to employees if not for all
            if (!$notification->for_all_employees && $request->has('employees')) {
                foreach ($request->employees as $employee_id) {
                    \DB::table('employee_notifications')->insert([
                        'notification_id' => $notification->id,
                        'employee_id' => $employee_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
            
            DB::commit();
            
    
            return redirect()->route('admin.notifications')->with('success', 'Notification created successfully!');
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
    }


    public function deleteNotification(Request $request){
        try {
            $notification = Notification::find($request->id);
            if(isset($notification)){
                
                DB::beginTransaction();
                $notification->delete();
                
                DB::commit();
    
                return redirect()->route('admin.notifications')->with('success','Success! notification deleted Successfully.');
            }else{
                return redirect()->route("admin.notifications")->with("error","Oops! Something Went Wrong. Try Again!");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Une Erreur est survenue, Veuillez réessayer plus tard!!!');
        }
        
    }






    
    
    




    public function logs(){
        $logs = Log::orderBy('id','desc')->get();
        return view('admin.logs-report',compact('logs'));
    }
    public function logsByUser(Request $request){
        $id = $request->id;
        $logs = Log::where('user_id',$id)->orderBy('id','desc')->get();
        return view('admin.logs-report',compact('logs'));
    }
    public function deleteLog(Request $request){
        $log = Log::find($request->id);
        if(isset($log)){
            $log->delete();
            return redirect()->route('admin.logs')->with('success','Success! Log deleted Successfully.');
        }else{
            return redirect()->route('admin.logs')->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }


    public function logout()
    {
        $user = User::find(session('current_admin')->id);
        $log = new Log();
        $log->details = "Admin: ".$user->name." logout";
        $log->user_id = $user->id;
        $log->save();

        session()->forget("current_admin");

        return redirect()->route('admin.login')->with('success','Success! Logged Out Successfully.');
    }

    public function messages(){
        return view ('admin.messages');
    }
    
    public function addMessage(Request $request){
        $user = User::find($request->id);
        $current_user = session('current_admin')->id;
        if(!isset($user) || $request->id == $current_user ){
            return redirect('/')->with('error','Oops! Something Went Wrong.');
        }else{
            return view('admin.add-Message',compact('user'));
        }
    }
    
    public function editProfile(){
        $user = User::find(session('current_admin')->id);
        return view('admin.edit-profile',compact('user'));
    }
    
    public function updateProfile(Request $request){
        $request->validate([
            'name' => 'required',
        ]);
        try {
            $user = User::find(session('current_admin')->id);
            if(isset($user)){
                DB::beginTransaction();


                $user->name = $request->name;
                $user->prenom = $request->prenom;
                
                // if(isset($request->picture)){
                //     $file = $request->file('picture');
                //     $file_name = "AdminProfile-".md5(microtime()).'.'.$file->getClientOriginalExtension();
                //     $destinationpath = 'layout/assets/avatars';
                //     $file->move($destinationpath,$file_name);
                //     $user->image = $file_name;
                // }

                if(isset($request->password)){
                    $user->password = $request->password;
                }
                
                $user->save();
                
                $log = new Log();
                $log->details = "Admin: ".$user->name." Edited his profile";
                $log->user_id = $user->id;
                $log->save();
                
                DB::commit();
            }
            return redirect()->route('admin.edit-profile')->with('success','Success! Profile Updated Successfully.');
        
        } catch (\Exception $e) {
            \Log::error('Error occurred: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred');
        }
    }
    public function generateLogs(Request $request){
        $user = User::find($request->id);
        if(isset($user)){
            $logs = Log::where('user_id',$user->id)->get();
            return view('admin.generate-logs',compact('logs'));
        }else{
            return redirect()->route("admin.logs")->with("error","Oops! Something Went Wrong. Try Again!");
        }
    }
    




    public function timelineChantier(Request $request)
    {
        $employees = User::where('role', 'employee')->get();
        
        // Eager load taches when getting chantiers
        $chantiers = Chantier::with('taches')->get();

        $metiers = Metier::pluck('nom', 'id'); // 👈 Fetch all metiers once, as [id => nom]

        
        $events = [];

        foreach ($chantiers as $chantier) {
            foreach ($chantier->taches as $tache) {

                // Decode the employees JSON array
                $tacheEmployeeIds = $tache->employees ?? [];

                // Fetch employee models for the task
                $tacheEmployees = User::whereIn('id', $tacheEmployeeIds)->get(['id', 'name', 'prenom', 'primary_metiers']);

                // Format employees with their primary_metiers
                $formattedEmployees = $tacheEmployees->map(function($employee) {
                    return [
                        'id' => $employee->id,
                        'nom' => $employee->name . ' ' . $employee->prenom,
                        'primary_metiers' => json_decode($employee->primary_metiers, true) ?? [],
                    ];
                });
                

                $formattedEmployees = $tacheEmployees->map(function($employee) use ($metiers) {
                    // Decode the primary_metiers
                    $primaryMetiersIds = json_decode($employee->primary_metiers, true) ?? [];
        
                    // Map IDs to their names
                    $primaryMetiersNames = collect($primaryMetiersIds)->map(function($id) use ($metiers) {
                        return $metiers[$id] ?? null; // If id exists, return its name
                    })->filter()->values()->toArray(); // remove nulls


                    // \Log::error('metiers ' . implode(', ', $primaryMetiersNames));  // Join array as a comma-separated string

        
                    return [
                        'id' => $employee->id,
                        'nom' => $employee->name." ".$employee->prenom,
                        'primary_metiers' => $primaryMetiersNames, // 👈 Now names, not IDs
                    ];
                });

                $endDate = new DateTime($chantier->date_de_fin);
                $endDate->modify('+1 day');
                $endDate = $endDate->format('Y-m-d');
                
                $events[] = [
                    'id' => $chantier->id,
                    'title' =>  $tache->nom,//$chantier->nom . ' - ' .
                    'start' => $chantier->date_de_debut,
                    'end' => $endDate,
                    'resourceId' => $chantier->id,
                    'extendedProps' => [
                        'tache_start' => $tache->date_debut,
                        'tache_end' => $tache->date_fin,
                        'chantier_id' => $chantier->id,
                        'chantier_nom' => $chantier->nom,
                        'tache_id' => $tache->id,
                        'tache_nom' => $tache->nom,
                        'color' => $this->getStatusColor($tache->status),
                        'employees' => $formattedEmployees, // 👈 Now has full employees info
                        'nb_jours' => $tache->nb_jours
                    ]
                ];
            }
        }

        return view('admin.timeline-chantier', compact('employees', 'events', 'chantiers'));
    }

    
    


    // public function timelineChantier(Request $request)
    // {
        
    
    //     return view('admin.timeline-chantier');
            
        
        
    // }
    

















    public function storeTimelineChantier(Request $request)
    {
        try {
            // Validate the request
            // $request->validate([
            //     'chantier.nom' => 'required|string|max:255',
            //     'chantier.date_de_debut' => 'required|date',
            //     'chantier.date_de_fin' => 'required|date|after_or_equal:chantier.date_de_debut',
            //     'chantier.taches' => 'nullable|array',
            // ]);

            // Log the incoming data for debugging
            \Log::info('New Chantier Data Received:', [
                'chantier' => $request->input('chantier'),
            ]);

            // // Begin transaction
            DB::beginTransaction();

            // Create the chantier
            $chantier = new Chantier();
            $chantier->nom = $request->input('chantier.nom');
            $chantier->adresse = $request->input('chantier.adresse');
            $chantier->date = Date('Y-m-d');
            $chantier->date_de_debut = $request->input('chantier.date_de_debut');
            $chantier->date_de_fin = $request->input('chantier.date_de_fin');
            $chantier->nbjours = $this->calculateWorkingDays($request->input('chantier.date_de_debut'), $request->input('chantier.date_de_fin'));
            $chantier->save();

            // Create the tasks if provided
            $events = [];
            if ($request->has('chantier.taches')) {
                foreach ($request->input('chantier.taches') as $tacheData) {
                    // Create the task
                    $tache = new Tache();
                    $tache->chantier_id = $chantier->id;
                    $tache->nom = $tacheData['nom'];
                    $tache->description = $tacheData['description'] ?? null;
                    $tache->date_debut = $tacheData['date_debut'];
                    $tache->date_fin = $tacheData['date_fin'];
                    $tache->status = $tacheData['status'] ?? 'pending';
                    $tache->employees = isset($tacheData['employees']) ? json_encode($tacheData['employees']) : null;
                    $tache->save();

                    // If there are employees, associate them with the task
                    if (!empty($tacheData['employees'])) {
                        $employees = User::whereIn('id', $tacheData['employees'])->get();
                        
                        // Create an event for the calendar
                        $employeesList = [];
                        foreach ($employees as $employee) {
                            $metierNames = [];
                            if ($employee->primary_metiers) {
                                $metierIds = json_decode($employee->primary_metiers, true);
                                foreach ($metierIds as $id) {
                                    $metier = \App\Models\Metier::find($id);
                                    if ($metier) {
                                        $metierNames[] = $metier->nom;
                                    }
                                }
                            }
                            
                            $employeesList[] = [
                                'id' => $employee->id,
                                'nom' => $employee->name . ' ' . $employee->prenom,
                                'primary_metiers' => $metierNames
                            ];
                        }
                        
                        $events[] = [
                            'id' => $tache->id,
                            'resourceId' => $chantier->id,
                            'title' => $tache->nom,
                            'start' => $tache->date_debut,
                            'end' => $tache->date_fin,
                            'extendedProps' => [
                                'employees' => $employeesList,
                                'color' => '#025182'
                            ]
                        ];
                    }
                }
            }

            // // Commit transaction
            DB::commit();

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Chantier créé avec succès',
                'chantier' => $chantier,
                'events' => $events
            ]);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error creating chantier:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Une erreur est survenue lors de la création du chantier',
                'errors' => [
                    'error' => [$e->getMessage()]
                ]
            ], 500);
        }
    }

    
    public function updateTimelineChantier(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'chantier.id' => 'required|exists:chantiers,id',
                'chantier.nom' => 'required|string|max:255',
                'chantier.date_de_debut' => 'required|date',
                'chantier.date_de_fin' => 'required|date|after_or_equal:chantier.date_de_debut',
                'chantier.taches' => 'nullable|array',
            ]);

            // Log the incoming data for debugging
            \Log::info('Update Chantier Data Received:', [
                'chantier' => $request->input('chantier'),
            ]);

            // Begin transaction
            DB::beginTransaction();

            // Find and update the chantier
            $chantier = Chantier::findOrFail($request->input('chantier.id'));
            $chantier->nom = $request->input('chantier.nom');
            $chantier->adresse = $request->input('chantier.adresse');
            $chantier->date_de_debut = $request->input('chantier.date_de_debut');
            $chantier->date_de_fin = $request->input('chantier.date_de_fin');
            $chantier->nbjours = $this->calculateWorkingDays($request->input('chantier.date_de_debut'), $request->input('chantier.date_de_fin'));
            $chantier->save();

            // Update tasks
            $events = [];
            if ($request->has('chantier.taches')) {
                // Get all current task IDs for this chantier
                $existingTaskIds = Tache::where('chantier_id', $chantier->id)->pluck('id')->toArray();
                $updatedTaskIds = [];
                
                foreach ($request->input('chantier.taches') as $tacheData) {
                    if (isset($tacheData['id']) && $tacheData['id']) {
                        // Update existing task
                        $tache = Tache::findOrFail($tacheData['id']);
                        $updatedTaskIds[] = $tache->id;
                    } else {
                        // Create new task
                        $tache = new Tache();
                        $tache->chantier_id = $chantier->id;
                    }
                    
                    // Update task data
                    $tache->nom = $tacheData['nom'];
                    $tache->description = $tacheData['description'] ?? null;
                    $tache->date_debut = $tacheData['date_debut'];
                    $tache->date_fin = $tacheData['date_fin'];
                    $tache->status = $tacheData['status'] ?? 'pending';
                    $tache->employees = isset($tacheData['employees']) ? json_encode($tacheData['employees']) : null;
                    $tache->save();
                    
                    // If there are employees, get their details for the event
                    if (!empty($tacheData['employees'])) {
                        $employees = User::whereIn('id', $tacheData['employees'])->get();
                        
                        // Create employee list for the event
                        $employeesList = [];
                        foreach ($employees as $employee) {
                            $metierNames = [];
                            if ($employee->primary_metiers) {
                                $metierIds = json_decode($employee->primary_metiers, true);
                                foreach ($metierIds as $id) {
                                    $metier = Metier::find($id);
                                    if ($metier) {
                                        $metierNames[] = $metier->nom;
                                    }
                                }
                            }
                            
                            $employeesList[] = [
                                'id' => $employee->id,
                                'nom' => $employee->name . ' ' . $employee->prenom,
                                'primary_metiers' => $metierNames
                            ];
                        }
                        
                        // Add to events array
                        $events[] = [
                            'id' => $tache->id,
                            'resourceId' => $chantier->id,
                            'title' => $tache->nom,
                            'start' => $tache->date_debut,
                            'end' => $tache->date_fin,
                            'extendedProps' => [
                                'employees' => $employeesList,
                                'color' => '#025182'
                            ]
                        ];
                    }
                }
                
                // Delete tasks that were not included in the update
                $tasksToDelete = array_diff($existingTaskIds, $updatedTaskIds);
                if (!empty($tasksToDelete)) {
                    Tache::whereIn('id', $tasksToDelete)->delete();
                }
            }

            // Commit transaction
            DB::commit();

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Chantier mis à jour avec succès',
                'chantier' => $chantier,
                'events' => $events
            ]);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error updating chantier:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Une erreur est survenue lors de la mise à jour du chantier',
                'errors' => [
                    'error' => [$e->getMessage()]
                ]
            ], 500);
        }
    }

    /**
     * Delete the specified chantier.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function deleteTimelineChantier(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'chantier_id' => 'required|exists:chantiers,id',
            ]);

            // Log the deletion request
            \Log::info('Delete Chantier Request:', [
                'chantier_id' => $request->input('chantier_id'),
            ]);

            // Begin transaction
            DB::beginTransaction();

            // Delete all tasks associated with this chantier
            Tache::where('chantier_id', $request->input('chantier_id'))->delete();
            
            // Delete the chantier
            $chantier = Chantier::findOrFail($request->input('chantier_id'));
            $chantier->delete();

            // Commit transaction
            DB::commit();

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Chantier supprimé avec succès',
            ]);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error deleting chantier:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Une erreur est survenue lors de la suppression du chantier',
                'errors' => [
                    'error' => [$e->getMessage()]
                ]
            ], 500);
        }
    }


    public function timelineEmployees(Request $request)
    {
        $metiers = Metier::pluck('nom', 'id');
        // $employees = User::where('role', 'employee')->get();
        $employees = User::where('role', 'employee')
            ->orderBy('display_order', 'asc')
            ->orderBy('name', 'asc') // Fallback
            ->get();
        $chantiers = Chantier::all();

        $resources = [];
        $events = [];

        // Build resources array for employees
        foreach ($employees as $employee) {
            $employeeMetiersIds = json_decode($employee->primary_metiers, true) ?? [];
            $employeeMetiers = collect($employeeMetiersIds)
                ->map(fn($id) => $metiers[$id] ?? null)
                ->filter()
                ->values()
                ->toArray();
        
            $resources[] = [
                'id' => $employee->id,
                'title' => $employee->name . ' ' . $employee->prenom,
                'avatar' => $employee->photo ?? null,
                'metiers' => $employeeMetiers,
                'order' => $employee->display_order ?? 999 // Add this line
            ];
        }

        // Fetch events with related data
        $eventRecords = Event::with(['chantier', 'employee'])
            ->whereHas('employee', function($query) {
                $query->where('role', 'employee');
            })
            ->get();

        // Build events array for FullCalendar
        foreach ($eventRecords as $eventRecord) {
            // Calculate end date (add 1 day for FullCalendar exclusive end)
            $endDate = (new DateTime($eventRecord->date_fin))
                ->modify('+1 day')
                ->format('Y-m-d');

            // Get taches for this event (assuming taches_id is a JSON array)
            $tachesIds = json_decode($eventRecord->taches_id, true) ?? [];
            $taches = [];
            $tacheNames = [];
            
            if (!empty($tachesIds)) {
                $taches = Tache::whereIn('id', $tachesIds)->get();
                $tacheNames = $taches->pluck('nom')->toArray();
            }

            $events[] = [
                'id' => $eventRecord->id,
                'resourceId' => $eventRecord->employee_id,
                'title' => $eventRecord->name . ($eventRecord->chantier && !empty($eventRecord->chantier->nom) ? ' - ' . $eventRecord->chantier->nom : ''),
                'start' => $eventRecord->date_debut,
                'end' => $endDate,
                'allDay' => true,
                'backgroundColor' => $eventRecord->event_color ?? '#3788d8',
                'borderColor' => $eventRecord->event_color ?? '#3788d8',
                'extendedProps' => [
                    'event_id' => $eventRecord->id ?? '',
                    'chantier_id' => $eventRecord->chantier->id ?? '',
                    'chantier_nom' => $eventRecord->chantier->nom ?? '',
                    'chantier_date_de_debut' => $eventRecord->chantier->date_de_debut ?? '',
                    'chantier_date_de_fin' => $eventRecord->chantier->date_de_fin ?? '',
                    'chantier_nbjours' => $eventRecord->chantier->nb_jours ?? '',
                    'taches_ids' => $tachesIds ?? '',
                    'taches_noms' => $tacheNames ?? '',
                    'taches_count' => count($taches) ?? '',
                    'taches_details' => collect($taches)->map(function($tache) {
                        return [
                            'id' => $tache->id,
                            'nom' => $tache->nom,
                            'description' => $tache->description,
                            'date_debut' => $tache->date_debut,
                            'date_fin' => $tache->date_fin,
                            'nb_jours' => $tache->nb_jours,
                            'status' => $tache->status,
                            'photo' => $tache->photo
                        ];
                    })->toArray(),
                    'employee_id' => $eventRecord->employee_id ?? '',
                    'employee_nom' => $eventRecord->employee->name . ' ' . $eventRecord->employee->prenom ?? '',
                    'name' => $eventRecord->name ?? '',
                    'note' => $eventRecord->note ?? '',
                    'status' => $eventRecord->status ?? '',
                    'date_debut' => $eventRecord->date_debut ?? '',
                    'date_fin' => $eventRecord->date_fin ?? '',
                    'created_at' => $eventRecord->created_at,
                    'updated_at' => $eventRecord->updated_at
                ]
            ];

        }


        // Add holidays and absences data
        $holidays = Holiday::where('is_active', true)
        ->whereYear('date_debut', '>=', date('Y'))
        ->get();

        $absences = Absence::with(['employee', 'chantier', 'tache'])
            // ->where('statut', 'approved')
            ->get();

        // Add to existing events array
        foreach ($holidays as $holiday) {
            $events[] = [
                'id' => 'holiday-' . $holiday->id,
                'title' => $holiday->name,
                'start' => $holiday->date_debut->format('Y-m-d'),
                'end' => $holiday->date_fin->addDay()->format('Y-m-d'),
                'allDay' => true,
                'display' => 'background',
                'backgroundColor' => '#ff9800',
                'borderColor' => '#ff9800',
                'classNames' => ['holiday-event'],
                'extendedProps' => [
                    'type' => 'holiday',
                    'holiday_id' => $holiday->id,
                    'editable' => false
                ]
            ];
        }

        foreach ($absences as $absence) {
            $events[] = [
                'id' => 'absence-' . $absence->id,
                'resourceId' => $absence->employee_id,
                'title' => $this->getAbsenceTitle($absence),
                'start' => $absence->date_debut,
                'end' => (new DateTime($absence->date_fin))->modify('+1 day')->format('Y-m-d'),
                'allDay' => true,
                'backgroundColor' => $this->getAbsenceColor($absence->type),
                'borderColor' => $this->getAbsenceColor($absence->type),
                'classNames' => ['absence-event'],
                'extendedProps' => [
                    'type' => 'absence',
                    'absence_id' => $absence->id,
                    'absence_type' => $absence->type,
                    'absence_statut' => $absence->statut,
                    'absence_reason' => $absence->reason,
                    'category' => $absence->category,
                    'editable' => true
                ]
            ];
        }

        usort($resources, function($a, $b) {
            return ($a['order'] ?? 999) <=> ($b['order'] ?? 999);
        });


        // \Log::info('Events:', $events);

        return view('admin.timeline-employees', compact('employees', 'resources', 'events', 'chantiers', 'holidays'));
    }

    public function getChantierLastColor(Request $request)
    {
        try {
            $chantierId = $request->input('chantier_id');
            
            // Find the most recent event for this chantier
            $lastEvent = Event::where('chantier_id', $chantierId)
                ->whereNotNull('event_color')
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($lastEvent && $lastEvent->event_color) {
                return response()->json([
                    'success' => true,
                    'color' => $lastEvent->event_color
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'No previous color found for this chantier'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching chantier color: ' . $e->getMessage()
            ], 500);
        }
    }

    public function reorderEmployees(Request $request)
    {
        try {
            $draggedEmployeeId = $request->dragged_employee_id;
            $targetEmployeeId = $request->target_employee_id;
            
            \Log::info('Reorder request:', [
                'dragged' => $draggedEmployeeId,
                'target' => $targetEmployeeId
            ]);
            
            // Start a database transaction
            DB::beginTransaction();
            
            $draggedEmployee = User::find($draggedEmployeeId);
            $targetEmployee = User::find($targetEmployeeId);
            
            if (!$draggedEmployee || !$targetEmployee) {
                DB::rollback();
                return response()->json([
                    'success' => false,
                    'message' => 'Employé non trouvé'
                ]);
            }
            
            // Get all employees ordered by current display_order
            $allEmployees = User::where('role', 'employee')
                ->orderBy('display_order', 'asc')
                ->orderBy('id', 'asc')
                ->get();
            
            \Log::info('Current order before change:', $allEmployees->pluck('name', 'id')->toArray());
            \Log::info('Current display_order values:', $allEmployees->pluck('display_order', 'name')->toArray());
            
            // Create new order array
            $newOrderedEmployees = [];
            $draggedAdded = false;
            
            foreach ($allEmployees as $employee) {
                // Skip the dragged employee in its original position
                if ($employee->id == $draggedEmployeeId) {
                    continue;
                }
                
                // If this is the target employee, add the dragged employee BEFORE it
                if ($employee->id == $targetEmployeeId && !$draggedAdded) {
                    $newOrderedEmployees[] = $draggedEmployee;
                    $draggedAdded = true;
                }
                
                $newOrderedEmployees[] = $employee;
            }
            
            // If dragged employee wasn't added yet (target was last), add it at the end
            if (!$draggedAdded) {
                $newOrderedEmployees[] = $draggedEmployee;
            }
            
            // Update display_order for all employees with explicit error checking
            $updateResults = [];
            foreach ($newOrderedEmployees as $index => $employee) {
                $newDisplayOrder = ($index + 1) * 10;
                
                $result = DB::table('users')
                    ->where('id', $employee->id)
                    ->update(['display_order' => $newDisplayOrder]);
                
                $updateResults[$employee->name] = [
                    'id' => $employee->id,
                    'old_order' => $employee->display_order,
                    'new_order' => $newDisplayOrder,
                    'update_result' => $result
                ];
            }
            
            \Log::info('Update results:', $updateResults);
            
            // Verify the updates worked
            $verifyEmployees = User::where('role', 'employee')
                ->orderBy('display_order', 'asc')
                ->get();
            
            \Log::info('Order after database update:', $verifyEmployees->pluck('name', 'display_order')->toArray());
            
            // Commit the transaction
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Ordre mis à jour avec succès',
                'new_order' => collect($newOrderedEmployees)->pluck('id')->toArray(),
                'update_results' => $updateResults
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('Reorder error:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour: ' . $e->getMessage()
            ]);
        }
    }

    public function getResourcesTimeline()
    {
        try {
            $employees = User::where('role', 'employee')
                ->orderBy('display_order', 'asc')
                ->orderBy('name', 'asc')
                ->get();

            $resources = $employees->map(function ($employee) {
                $metierNames = [];
                if ($employee->primary_metiers) {
                    $metierIds = json_decode($employee->primary_metiers, true);
                    foreach ($metierIds as $id) {
                        $metier = \App\Models\Metier::find($id);
                        if ($metier) {
                            $metierNames[] = $metier->nom;
                        }
                    }
                }

                return [
                    'id' => (string) $employee->id,
                    'title' => $employee->name . ' ' . $employee->prenom,
                    'order' => $employee->display_order ?: 999,
                    'extendedProps' => [
                        'metier' => implode(', ', $metierNames),
                        'order' => $employee->display_order ?: 999,
                        'avatar' => $employee->avatar ?? null
                    ]
                ];
            });

            return response()->json([
                'success' => true,
                'resources' => $resources
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching resources:', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error fetching resources'
            ]);
        }
    }



    public function storeSelectEventEmployee(Request $request)
    {
        try {
            // Log the incoming data for debugging
            \Log::info('Employee Event Data Received:', $request->all());
    
            // Begin transaction
            DB::beginTransaction();
    
            $tachesIds = [];
    
            // Handle chantier creation or selection
            $chantier = null;

            if ($request->has('chantier_type')) {
                if ($request->input('chantier_type') === 'new' && 
                    $request->has('new_chantier') && 
                    is_array($request->input('new_chantier')) &&
                    !empty($request->input('new_chantier.nom'))) {
                    
                    $nom = $request->input('new_chantier.nom');
                    if (!empty($nom)) {
                        // Create new chantier
                        $chantierId = $this->createNewChantier($request->input('new_chantier'));
                        $chantier = Chantier::find($chantierId);
                    }
                } 
                elseif ($request->input('chantier_type') === 'existing' && 
                    $request->has('existing_chantier_id')) {
                    
                    $chantierId = $request->input('existing_chantier_id');
                    if (!empty($chantierId)) {
                        $chantier = Chantier::find($chantierId);
                        if (!$chantier) {
                            throw new \Exception('Chantier introuvable');
                        }
                    }
                }
            }

            // Later in your code, check if $chantier is not null before using it
            if ($chantier && $request->has('taches') && !empty($request->input('taches'))) {
                $tachesIds = $this->createTaches($request->input('taches'), $chantier->id);
            }
    
            // Create the event
            $event = new Event(); // Adjust the model path as needed
            $event->name = $request->input('event_name');
            if ($chantier) {
                $event->chantier_id = $chantier->id;
            }
            $event->taches_id = json_encode($tachesIds); // Store as JSON array
            // $event->employee_id = $request->input('resource_id'); // The selected employee
            $event->employee_id = $request->input('employee_id'); // The selected employee
            $event->date_debut = $request->input('event_start_date');
            $event->date_fin = $request->input('event_end_date');
            $event->note = $request->input('event_note');
            $event->event_color = $request->input('event_color', '#52357B');
            $event->status = 'planifie'; // Default status
            $event->save();

            if ($chantier && $request->input('chantier_type') === 'existing') {
                $newColor = $request->input('event_color', '#52357B');
                
                // Update all existing events for this chantier to use the new color
                Event::where('chantier_id', $chantier->id)
                    ->where('id', '!=', $event->id) // Exclude the current event we just created
                    ->update(['event_color' => $newColor]);
                
                \Log::info('Updated colors for existing events in chantier: ' . $chantier->id . ' to color: ' . $newColor);
            }
    
            // Commit transaction
            DB::commit();
    
            $event->load(['chantier', 'employee']);

            // Return success response with complete event data
            return response()->json([
                'success' => true,
                'message' => 'Événement créé avec succès',
                'event' => [
                    'id' => $event->id,
                    'name' => $event->name,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'note' => $event->note,
                    'event_color' => $event->event_color,
                    'employee_id' => $event->employee_id,
                    'chantier_id' => $event->chantier_id,
                    'taches_id' => $event->taches_id,
                    'status' => $event->status,
                    'created_at' => $event->created_at,
                    'updated_at' => $event->updated_at,
                    'employee' => $event->employee ? [
                        'id' => $event->employee->id,
                        'name' => $event->employee->name,
                        'prenom' => $event->employee->prenom
                    ] : null
                ],
                'chantier' => $chantier ? [
                    'id' => $chantier->id,
                    'nom' => $chantier->nom,
                    'adresse' => $chantier->adresse,
                    'date_de_debut' => $chantier->date_de_debut,
                    'date_de_fin' => $chantier->date_de_fin,
                    'nb_jours' => $chantier->nbjours
                ] : null,
                'taches_count' => count($tachesIds)
            ]);
    
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error creating employee event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            // Return error response
            return response()->json([
                'success' => false,
                'message' => 'Une erreur est survenue lors de la création de l\'événement'// . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Create a new chantier
     */
    private function createNewChantier($chantierId)
    {
        
        $chantier = new Chantier();
        $chantier->nom = $chantierId['nom'];
        $chantier->adresse = $chantierId['adresse'];
        $chantier->date = Date('Y-m-d');
        $chantier->date_de_debut = $chantierId['date_debut'];
        $chantier->date_de_fin = $chantierId['date_fin'];
        $chantier->nbjours = $this->calculateWorkingDays($chantierId['date_debut'], $chantierId['date_fin']);
        $chantier->save();
    
        return $chantier->id;
    
    }
    
    /**
     * Create tasks for the chantier
     */
    private function createTaches($tachesData, $chantierId)
    {
        $tachesIds = [];
    
        foreach ($tachesData as $tacheData) {
            if (empty($tacheData['nom'])) {
                continue; // Skip empty tasks
            }
    
            $tache = new Tache();
            $tache->chantier_id = $chantierId;
            $tache->nom = $tacheData['nom'];
            $tache->description = $tacheData['description'] ?? null;
            
            // Handle dates - use chantier dates if task dates are empty
            $tache->date_debut = !empty($tacheData['date_debut']) ? $tacheData['date_debut'] : null;
            $tache->date_fin = !empty($tacheData['date_fin']) ? $tacheData['date_fin'] : null;
            
            // Calculate working days if both dates are provided
            if ($tache->date_debut && $tache->date_fin) {
                $tache->nb_jours = $this->calculateWorkingDays($tache->date_debut, $tache->date_fin);
            }
            
            $tache->status = $tacheData['status'] ?? 'pending';
            
            // Initialize employees as empty array for now
            // You can modify this later to assign employees to tasks
            $tache->employees = json_encode([]);
            
            $tache->save();
            
            $tachesIds[] = $tache->id;
        }
    
        return $tachesIds;
    }
    
    public function updateEventEmployee(Request $request)
{
    try {
        \Log::info('Update Event Request:', $request->all());

        // Validate the request
        $request->validate([
            'event_id' => 'required|exists:events,id',
            'event_name' => 'required|string|max:255',
            'event_start_date' => 'required|date',
            'event_end_date' => 'required|date|after_or_equal:event_start_date',
            'edit-chantier' => 'nullable|exists:chantiers,id',
            'edit-chantier-date_de_debut' => 'nullable|date',
            'edit-chantier-date_de_fin' => 'nullable|date|after_or_equal:edit-chantier-date_de_debut',
            'employee_id' => 'required|exists:users,id',
        ]);

        // Begin transaction
        DB::beginTransaction();

        // Find the event
        $event = Event::findOrFail($request->input('event_id'));
        
        // Store the original color and chantier_id to check if they changed
        $originalColor = $event->event_color;
        $originalChantierId = $event->chantier_id;
        
        if($request->input('edit-chantier')){
            
            // Update the chantier dates if needed
            $chantier = Chantier::findOrFail($request->input('edit-chantier'));
            if($chantier){
                $chantier->date_de_debut = $request->input('edit-chantier-date_de_debut');
                $chantier->date_de_fin = $request->input('edit-chantier-date_de_fin');
                // $chantier->save();

                $event->chantier_id = $request->input('edit-chantier');
            }
        }

        // Update the event
        $event->name = $request->input('event_name');
        $event->employee_id = $request->input('employee_id');
        $event->date_debut = $request->input('event_start_date');
        $event->date_fin = $request->input('event_end_date');
        $event->note = $request->input('event_note');
        $event->event_color = $request->input('event_color', $event->event_color);
        $event->save();

        // Check if color changed and event is linked to a chantier
        $newColor = $request->input('event_color');
        if ($event->chantier_id && $newColor && $newColor !== $originalColor) {
            // Update all other events for this chantier to use the new color
            Event::where('chantier_id', $event->chantier_id)
                ->where('id', '!=', $event->id) // Exclude the current event we just updated
                ->update(['event_color' => $newColor]);
            
            \Log::info('Updated colors for existing events in chantier: ' . $event->chantier_id . ' to color: ' . $newColor);
        }

        // Handle taches
        if ($request->has('taches')) {
            // Get existing tache IDs from the event's taches_id column
            $existingTacheIds = json_decode($event->taches_id, true) ?? [];
            $updatedTacheIds = [];

            // Decode the JSON string to an array
            $taches = json_decode($request->taches, true);
            
            // Ensure we have a valid array
            if (!is_array($taches)) {
                $taches = [];
            }

            // Process each tache from the request
            foreach ($taches as $tacheData) {
                $tacheData = (array)$tacheData; // Ensure it's an array
                
                if (isset($tacheData['id']) && $tacheData['id']) {
                    // Update existing tache
                    $tache = Tache::find($tacheData['id']);
                    if ($tache) {
                        $tache->update([
                            'nom' => $tacheData['nom'],
                            'description' => $tacheData['description'] ?? null,
                            'date_debut' => !empty($tacheData['date_debut']) ? $tacheData['date_debut'] : null,
                            'date_fin' => !empty($tacheData['date_fin']) ? $tacheData['date_fin'] : null,
                            'nb_jours' => $this->calculateWorkingDays($tacheData['date_debut'], $tacheData['date_fin']),
                            'employees' => json_encode([$event->employee_id]),
                            'status' => $tacheData['status'],
                        ]);
                        $updatedTacheIds[] = (string)$tache->id;
                    }
                } else {
                    // Create new tache
                    $tache = Tache::create([
                        'nom' => $tacheData['nom'],
                        'description' => $tacheData['description'] ?? null,
                        'date_debut' => !empty($tacheData['date_debut']) ? $tacheData['date_debut'] : null,
                        'date_fin' => !empty($tacheData['date_fin']) ? $tacheData['date_fin'] : null,
                        'status' => $tacheData['status'],
                        'chantier_id' => $event->chantier_id,
                        'nb_jours' => $this->calculateWorkingDays($tacheData['date_debut'], $tacheData['date_fin']),
                        'employees' => json_encode([$event->employee_id]),
                    ]);
                    $updatedTacheIds[] = (string)$tache->id;
                }
            }

            // Update the event's taches_id with the new array
            $event->taches_id = json_encode($updatedTacheIds);
            $event->save();

            // Delete taches that were removed
            $tachesToDelete = array_diff($existingTacheIds, $updatedTacheIds);
            if (!empty($tachesToDelete)) {
                Tache::whereIn('id', $tachesToDelete)->delete();
            }
        }

        // Commit transaction
        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Événement mis à jour avec succès',
            'event' => $event,
            'chantier' => $chantier ?? ''
        ]);

    } catch (\Exception $e) {
        // Rollback transaction on error
        DB::rollBack();
        
        // Log the error
        \Log::error('Error updating event:', [
            'message' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'request_data' => $request->all()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Une erreur est survenue lors de la mise à jour de l\'événement: ' . $e->getMessage()
        ], 500);
    }
}



    public function deleteEventEmployee(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'event_id' => 'required|exists:events,id',
            ]);

            // Log the deletion request
            \Log::info('Delete Event Request:', [
                'event_id' => $request->input('event_id'),
            ]);

            // Begin transaction
            DB::beginTransaction();
            
            // Delete the event
            $event = Event::findOrFail($request->input('event_id'));
            $event->delete();

            // Commit transaction
            DB::commit();

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Evenement supprimé avec succès',
            ]);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error deleting Evenement:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Une erreur est survenue lors de la suppression de l\'Evenement',
                'errors' => [
                    'error' => [$e->getMessage()]
                ]
            ], 500);
        }
    }



    /**
     * Move an event to a new date/employee - FIXED VERSION
     */
    public function moveEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Move Event Request:', $request->all());
            
            $request->validate([
                'event_id' => 'required|exists:events,id',
                'new_start_date' => 'required|date',
                'new_end_date' => 'required|date|after_or_equal:new_start_date',
                'new_employee_id' => 'required|exists:users,id',
                'old_employee_id' => 'nullable|exists:users,id'
            ]);

            DB::beginTransaction();

            // Find the event
            $event = Event::findOrFail($request->input('event_id'));
            
            \Log::info('BEFORE Update - Event:', [
                'id' => $event->id,
                'name' => $event->name,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin,
                'employee_id' => $event->employee_id
            ]);
            
            // FIXED: Explicitly update each field and force save
            $event->date_debut = $request->input('new_start_date');
            $event->date_fin = $request->input('new_end_date');
            $event->employee_id = $request->input('new_employee_id');
            
            // Force save and check if it actually saved
            $saveResult = $event->save();
            
            \Log::info('Save result:', ['success' => $saveResult]);
            
            // Refresh from database to confirm changes
            $event->refresh();
            
            \Log::info('AFTER Update - Event from DB:', [
                'id' => $event->id,
                'name' => $event->name,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin,
                'employee_id' => $event->employee_id
            ]);
            
            // Also check the actual database record
            $dbCheck = DB::table('events')->where('id', $event->id)->first();
            \Log::info('Direct DB Check:', [
                'date_debut' => $dbCheck->date_debut,
                'date_fin' => $dbCheck->date_fin,
                'employee_id' => $dbCheck->employee_id
            ]);

            // Update associated taches with new employee if any
            if ($event->taches_id) {
                $tachesIds = json_decode($event->taches_id, true) ?? [];
                if (!empty($tachesIds)) {
                    $updateResult = Tache::whereIn('id', $tachesIds)->update([
                        'employees' => json_encode([$request->input('new_employee_id')])
                    ]);
                    \Log::info('Updated taches count:', $updateResult);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Événement déplacé avec succès',
                'event' => [
                    'id' => $event->id,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'employee_id' => $event->employee_id,
                    'name' => $event->name
                ],
                'debug' => [
                    'request_dates' => [
                        'start' => $request->input('new_start_date'),
                        'end' => $request->input('new_end_date')
                    ],
                    'saved_dates' => [
                        'start' => $event->date_debut,
                        'end' => $event->date_fin
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error moving event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du déplacement: ' . $e->getMessage(),
                'debug' => [
                    'request' => $request->all(),
                    'error' => $e->getMessage()
                ]
            ], 500);
        }
    }

    /**
     * Resize an event (change duration) - FIXED VERSION
     */
    public function resizeEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Resize Event Request:', $request->all());
            
            $request->validate([
                'event_id' => 'required|exists:events,id',
                'new_start_date' => 'required|date',
                'new_end_date' => 'required|date|after_or_equal:new_start_date'
            ]);

            DB::beginTransaction();

            // Find the event
            $event = Event::findOrFail($request->input('event_id'));
            
            \Log::info('BEFORE Resize - Event:', [
                'id' => $event->id,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin
            ]);
            
            // FIXED: Explicitly update fields
            $event->date_debut = $request->input('new_start_date');
            $event->date_fin = $request->input('new_end_date');
            
            // Force save
            $saveResult = $event->save();
            
            \Log::info('Resize save result:', ['success' => $saveResult]);
            
            // Refresh and verify
            $event->refresh();
            
            \Log::info('AFTER Resize - Event from DB:', [
                'id' => $event->id,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin
            ]);

            // Update associated taches dates if needed
            if ($event->taches_id) {
                $tachesIds = json_decode($event->taches_id, true) ?? [];
                if (!empty($tachesIds)) {
                    $updateCount = 0;
                    foreach ($tachesIds as $tacheId) {
                        $tache = Tache::find($tacheId);
                        if ($tache) {
                            $tache->date_debut = $request->input('new_start_date');
                            $tache->date_fin = $request->input('new_end_date');
                            $tache->nb_jours = $this->calculateWorkingDays(
                                $request->input('new_start_date'), 
                                $request->input('new_end_date')
                            );
                            $tache->save();
                            $updateCount++;
                        }
                    }
                    \Log::info('Updated taches count:', $updateCount);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Durée de l\'événement modifiée avec succès',
                'event' => [
                    'id' => $event->id,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'name' => $event->name
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error resizing event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du redimensionnement: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Duplicate an event - FIXED VERSION
     */
    public function duplicateEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Duplicate Event Request:', $request->all());
            
            $request->validate([
                'original_event_id' => 'required|exists:events,id',
                'duplicate_start_date' => 'required|date',
                'duplicate_end_date' => 'required|date|after_or_equal:duplicate_start_date',
                'duplicate_employee_id' => 'required|exists:users,id'
            ]);

            DB::beginTransaction();

            // Find the original event
            $originalEvent = Event::with(['chantier', 'employee'])->findOrFail($request->input('original_event_id'));
            
            \Log::info('Original Event:', [
                'id' => $originalEvent->id,
                'name' => $originalEvent->name,
                'date_debut' => $originalEvent->date_debut,
                'date_fin' => $originalEvent->date_fin,
                'employee_id' => $originalEvent->employee_id
            ]);
            
            // Create new event as a copy
            $newEvent = new Event();
            $newEvent->name = $originalEvent->name; // Frontend will add " (Copie)" to title
            $newEvent->chantier_id = $originalEvent->chantier_id;
            $newEvent->employee_id = $request->input('duplicate_employee_id');
            $newEvent->date_debut = $request->input('duplicate_start_date');
            $newEvent->date_fin = $request->input('duplicate_end_date');
            $newEvent->note = $originalEvent->note;
            $newEvent->event_color = $originalEvent->event_color;
            $newEvent->status = 'planifie'; // Reset status for duplicated event
            
            // Duplicate taches if any
            $newTachesIds = [];
            if ($originalEvent->taches_id) {
                $originalTachesIds = json_decode($originalEvent->taches_id, true) ?? [];
                
                \Log::info('Duplicating taches:', $originalTachesIds);
                
                foreach ($originalTachesIds as $tacheId) {
                    $originalTache = Tache::find($tacheId);
                    if ($originalTache) {
                        $newTache = new Tache();
                        $newTache->chantier_id = $originalTache->chantier_id;
                        $newTache->nom = $originalTache->nom;
                        $newTache->description = $originalTache->description;
                        
                        // Use the new event dates for the duplicated taches
                        $newTache->date_debut = $request->input('duplicate_start_date');
                        $newTache->date_fin = $request->input('duplicate_end_date');
                        $newTache->nb_jours = $this->calculateWorkingDays(
                            $request->input('duplicate_start_date'), 
                            $request->input('duplicate_end_date')
                        );
                        $newTache->status = 'pending'; // Reset status
                        $newTache->employees = json_encode([$request->input('duplicate_employee_id')]);
                        
                        $newTache->save();
                        $newTachesIds[] = $newTache->id;
                    }
                }
            }
            
            $newEvent->taches_id = json_encode($newTachesIds);
            $newEvent->save();
            
            \Log::info('New Event Created:', [
                'id' => $newEvent->id,
                'name' => $newEvent->name,
                'date_debut' => $newEvent->date_debut,
                'date_fin' => $newEvent->date_fin,
                'employee_id' => $newEvent->employee_id,
                'taches_count' => count($newTachesIds)
            ]);

            // Load relationships for response
            $newEvent->load(['chantier', 'employee']);
            
            // Get taches details for response
            $tachesDetails = [];
            if (!empty($newTachesIds)) {
                $taches = Tache::whereIn('id', $newTachesIds)->get();
                $tachesDetails = $taches->map(function($tache) {
                    return [
                        'id' => $tache->id,
                        'nom' => $tache->nom,
                        'description' => $tache->description,
                        'date_debut' => $tache->date_debut,
                        'date_fin' => $tache->date_fin,
                        'nb_jours' => $tache->nb_jours,
                        'status' => $tache->status,
                        'photo' => $tache->photo
                    ];
                })->toArray();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Événement dupliqué avec succès',
                'event' => [
                    'id' => $newEvent->id,
                    'name' => $newEvent->name,
                    'date_debut' => $newEvent->date_debut,
                    'date_fin' => $newEvent->date_fin,
                    'note' => $newEvent->note,
                    'event_color' => $newEvent->event_color,
                    'employee_id' => $newEvent->employee_id,
                    'chantier_id' => $newEvent->chantier_id,
                    'taches_id' => $newEvent->taches_id,
                    'status' => $newEvent->status,
                    'created_at' => $newEvent->created_at,
                    'updated_at' => $newEvent->updated_at,
                    'employee' => $newEvent->employee ? [
                        'id' => $newEvent->employee->id,
                        'name' => $newEvent->employee->name,
                        'prenom' => $newEvent->employee->prenom
                    ] : null
                ],
                'chantier' => $newEvent->chantier ? [
                    'id' => $newEvent->chantier->id,
                    'nom' => $newEvent->chantier->nom,
                    'adresse' => $newEvent->chantier->adresse,
                    'date_de_debut' => $newEvent->chantier->date_de_debut,
                    'date_de_fin' => $newEvent->chantier->date_de_fin,
                    'nb_jours' => $newEvent->chantier->nbjours
                ] : null,
                'taches_details' => $tachesDetails
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error duplicating event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la duplication: ' . $e->getMessage(),
                'debug' => [
                    'request' => $request->all(),
                    'error' => $e->getMessage()
                ]
            ], 500);
        }
    }


    private function getAbsenceTitle($absence)
    {
        $titles = [
            'absence' => 'Autre',
            'conge_individuel' => 'Congé payé',
            'conge_collectif' => 'Congé collectif',
            'conge_sans_solde' => 'Congé sans solde',
            'absence_injustifiee' => 'Absence injustifiée',
            'absence_justifiee' => 'Absence justifiée',
            'arret_maladie' => 'Arrêt maladie'
        ];
        
        $title = $titles[$absence->type] ?? 'Autre';
        if ($absence->category && $absence->category !== $title) {
            $title .= ' - ' . $absence->category;
        }
        
        return $title;
    }

    private function getAbsenceColor($type)
    {
        return match($type) {
            'absence' => '#607d8b',           // Blue Grey
            'conge_individuel' => '#4caf50',  // Green
            'conge_collectif' => '#ff9800',   // Orange
            default => '#607d8b'
        };
    }

    


    public function storeAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'type' => 'required|in:absence,conge_individuel,conge_collectif',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'category' => 'nullable|string',
            'reason' => 'nullable|string',
            'statut' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::create([
                'employee_id' => $request->employee_id,
                'type' => $request->type,
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'category' => 'Autre',
                'reason' => $request->reason,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin),
                'statut' => $request->statut
                // 'statut' => $request->type === 'conge_collectif' ? 'approved' : 'pending'
            ]);

            DB::commit();

            // Return data in timeline format
            return response()->json([
                'success' => true,
                'absence' => [
                    'id' => $absence->id,
                    'employee_id' => $absence->employee_id,
                    'type' => $absence->type,
                    'date_debut' => $absence->date_debut,
                    'date_fin' => $absence->date_fin,
                    'category' => $absence->category,
                    'reason' => $absence->reason,
                    'statut' => $absence->statut
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la création: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update absence from timeline (drag/drop)
     */
    public function updateAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'employee_id' => 'required|exists:users,id',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'type' => 'nullable|in:absence,conge_individuel,conge_collectif',
            'category' => 'nullable|string',
            'reason' => 'nullable|string',
            'statut' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $updateData = [
                'employee_id' => $request->employee_id,
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin)
            ];

            if ($request->has('type')) $updateData['type'] = $request->type;
            if ($request->has('reason')) $updateData['reason'] = $request->reason;
            if ($request->has('statut')) $updateData['statut'] = $request->statut;

            $absence->update($updateData);
            
            DB::commit();

            return response()->json([
                'success' => true,
                'absence' => $absence->fresh()
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete absence from timeline
     */
    public function deleteAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id'
        ]);

        try {
            DB::beginTransaction();
            
            $absence = Absence::findOrFail($request->absence_id);
            $absence->delete();
            
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Absence supprimée avec succès'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la suppression: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Move absence (drag and drop)
     */
    public function moveAbsence(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'new_start_date' => 'required|date',
            'new_end_date' => 'required|date|after_or_equal:new_start_date',
            'new_employee_id' => 'required|exists:users,id'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $absence->update([
                'employee_id' => $request->new_employee_id,
                'date_debut' => $request->new_start_date,
                'date_fin' => $request->new_end_date,
                'jours_absence' => $this->calculateWorkingDays($request->new_start_date, $request->new_end_date)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Absence déplacée avec succès'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du déplacement: ' . $e->getMessage()
            ], 500);
        }
    }

    public function resizeAbsence(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $absence->update([
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Durée modifiée avec succès',
                'absence' => $absence->fresh()
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error resizing absence:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du redimensionnement: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get default category based on type
     */
    private function getDefaultCategory($type)
    {
        return match($type) {
            'absence' => 'Autre',
            'conge_individuel' => 'Congé individuel',
            'conge_collectif' => 'Congé collectif',
            // 'conge_sans_solde' => 'Congé sans solde',
            // 'absence_injustifiee' => 'Absence injustifiée',
            // 'absence_justifiee' => 'Absence justifiée',
            // 'arret_maladie' => 'Arrêt maladie',
            default => 'Autre'
        };
    }




    public function settings()
    {
        $holidays = Holiday::orderBy('date_debut')->get();
        return view('admin.settings', compact('holidays'));
    }

    public function storeHoliday(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'type' => 'required|in:national,regional,custom'
        ]);

        Holiday::create($request->all());
        
        return redirect()->route('admin.settings')->with('success', 'Jour férié ajouté avec succès');
    }

    public function toggleHoliday($id)
    {
        $holiday = Holiday::findOrFail($id);
        $holiday->is_active = !$holiday->is_active;
        $holiday->save();
        
        return redirect()->route('admin.settings')->with('success', 'Statut du jour férié modifié');
    }


    public function updateHoliday(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'type' => 'required|in:national,regional,custom',
            'region' => 'nullable|string'
        ], [
            'date_fin.after_or_equal' => 'La date de fin doit être égale ou postérieure à la date de début'
        ]);

        $holiday = Holiday::findOrFail($id);
        $holiday->update($request->all());
        
        return redirect()->route('admin.settings')->with('success', 'Jour férié mis à jour avec succès');
    }

    public function deleteHoliday($id)
    {
        Holiday::findOrFail($id)->delete();
        
        return response()->json(['success' => true]);
    }

    public function elementRequests()
    {
        $requests = ElementRequest::with(['employee', 'element'])
                                ->orderBy('created_at', 'desc')
                                ->get();
        return view('admin.element-requests', compact('requests'));
    }

    public function approveElementRequest($id)
    {
        try {
            DB::beginTransaction();
            
            $request = ElementRequest::find($id);
            
            if (!$request || $request->status !== 'pending') {
                return redirect()->back()->with('error', 'Demande non trouvée ou déjà traitée.');
            }
            
            $element = Element::find($request->element_id);
            
            if ($element->quantity < $request->quantity_requested) {
                return redirect()->back()->with('error', 'Stock insuffisant pour approuver cette demande.');
            }
            
            // Update element quantity
            $element->quantity -= $request->quantity_requested;
            $element->save();
            
            // Update request status
            $request->status = 'approved';
            $request->save();
            
            DB::commit();
            
            return redirect()->back()->with('success', 'Demande approuvée avec succès.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Une erreur est survenue.');
        }
    }

    public function rejectElementRequest(Request $request)
    {
        $elementRequest = ElementRequest::find($request->request_id);
        
        if (!$elementRequest || $elementRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Demande non trouvée ou déjà traitée.');
        }
        
        $elementRequest->status = 'rejected';
        $elementRequest->admin_notes = $request->admin_notes;
        $elementRequest->save();
        
        return redirect()->back()->with('success', 'Demande rejetée.');
    }
    


    public function materielRequests()
    {
        $requests = MaterielRequest::with(['employee', 'materiel'])
                                ->orderBy('created_at', 'desc')
                                ->get();
        return view('admin.materiel-requests', compact('requests'));
    }

    public function approveMaterielRequest($id)
    {
        try {
            DB::beginTransaction();
            
            $request = MaterielRequest::find($id);
            
            if (!$request || $request->status !== 'pending') {
                return redirect()->back()->with('error', 'Demande non trouvée ou déjà traitée.');
            }
            
            $materiel = Materiel::find($request->materiel_id);
            
            if ($materiel->status !== 'disponible') {
                return redirect()->back()->with('error', 'Ce matériel n\'est plus disponible.');
            }
            
            // Update materiel status to indisponible
            $materiel->status = 'indisponible';
            $materiel->save();
            
            // Update request status
            $request->status = 'approved';
            $request->save();

            
            // Create assignment record
            MaterielSalarie::create([
                'materiel_id' => $materiel->id,
                'salarie_id' => $request->employee->id,
                'date_de_prise' => now(),
                'date_de_remise' => $request->date_needed ?? now()->addDays(30),
                'status' => 'assigned'
            ]);
            
            DB::commit();
            
            return redirect()->back()->with('success', 'Demande approuvée et matériel affecté avec succès.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Une erreur est survenue.');
        }
    }

    public function rejectMaterielRequest(Request $request)
    {
        $materielRequest = MaterielRequest::find($request->request_id);
        
        if (!$materielRequest || $materielRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Demande non trouvée ou déjà traitée.');
        }
        
        $materielRequest->status = 'rejected';
        $materielRequest->admin_notes = $request->admin_notes;
        $materielRequest->save();
        
        return redirect()->back()->with('success', 'Demande rejetée.');
    }

}
