<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Notification;
use Carbon\Carbon;
use App\Models\Message;
use DateTime;
use Carbon\CarbonPeriod;
use App\Models\Metier;
use App\Models\Chantier;
use App\Models\Tache;
use App\Models\Famille;
use App\Models\Lieu;
use App\Models\Element;
use App\Models\Materiel;
use App\Models\MaterielSalarie;
use App\Models\Absence;
use App\Models\Event;
use DateInterval;
use DatePeriod;
use App\Models\StockMovement;
use App\Models\Holiday;
use Session;

use App\Models\MaterielRequest;
use App\Models\ElementRequest;


class EmployeeController extends Controller
{   

    
    // public function dashboard(Request $request){
    //     $employee = User::find(session('current_employee')->id);

        
    //     return view('employee.dashboard',compact('employee'));
    // }


    public function dashboard(Request $request)
    {
        $metiers = Metier::pluck('nom', 'id');
        // $employees = User::where('role', 'employee')->get();
        $employees = User::where('role', 'employee')
            ->orderBy('display_order', 'asc')
            ->orderBy('name', 'asc') // Fallback
            ->get();
        $chantiers = Chantier::all();

        $resources = [];
        $events = [];

        // Build resources array for employees
        foreach ($employees as $employee) {
            $employeeMetiersIds = json_decode($employee->primary_metiers, true) ?? [];
            $employeeMetiers = collect($employeeMetiersIds)
                ->map(fn($id) => $metiers[$id] ?? null)
                ->filter()
                ->values()
                ->toArray();
        
            $resources[] = [
                'id' => $employee->id,
                'title' => $employee->name . ' ' . $employee->prenom,
                'avatar' => $employee->photo ?? null,
                'metiers' => $employeeMetiers,
                'order' => $employee->display_order ?? 999 // Add this line
            ];
        }

        // Fetch events with related data
        $eventRecords = Event::with(['chantier', 'employee'])
            ->whereHas('employee', function($query) {
                $query->where('role', 'employee');
            })
            ->get();

        // Build events array for FullCalendar
        foreach ($eventRecords as $eventRecord) {
            // Calculate end date (add 1 day for FullCalendar exclusive end)
            $endDate = (new DateTime($eventRecord->date_fin))
                ->modify('+1 day')
                ->format('Y-m-d');

            // Get taches for this event (assuming taches_id is a JSON array)
            $tachesIds = json_decode($eventRecord->taches_id, true) ?? [];
            $taches = [];
            $tacheNames = [];
            
            if (!empty($tachesIds)) {
                $taches = Tache::whereIn('id', $tachesIds)->get();
                $tacheNames = $taches->pluck('nom')->toArray();
            }

            $events[] = [
                'id' => $eventRecord->id,
                'resourceId' => $eventRecord->employee_id,
                'title' => $eventRecord->name . ($eventRecord->chantier && !empty($eventRecord->chantier->nom) ? ' - ' . $eventRecord->chantier->nom : ''),
                'start' => $eventRecord->date_debut,
                'end' => $endDate,
                'allDay' => true,
                'backgroundColor' => $eventRecord->event_color ?? '#3788d8',
                'borderColor' => $eventRecord->event_color ?? '#3788d8',
                'extendedProps' => [
                    'event_id' => $eventRecord->id ?? '',
                    'chantier_id' => $eventRecord->chantier->id ?? '',
                    'chantier_nom' => $eventRecord->chantier->nom ?? '',
                    'chantier_date_de_debut' => $eventRecord->chantier->date_de_debut ?? '',
                    'chantier_date_de_fin' => $eventRecord->chantier->date_de_fin ?? '',
                    'chantier_nbjours' => $eventRecord->chantier->nb_jours ?? '',
                    'taches_ids' => $tachesIds ?? '',
                    'taches_noms' => $tacheNames ?? '',
                    'taches_count' => count($taches) ?? '',
                    'taches_details' => collect($taches)->map(function($tache) {
                        return [
                            'id' => $tache->id,
                            'nom' => $tache->nom,
                            'description' => $tache->description,
                            'date_debut' => $tache->date_debut,
                            'date_fin' => $tache->date_fin,
                            'nb_jours' => $tache->nb_jours,
                            'status' => $tache->status,
                            'photo' => $tache->photo
                        ];
                    })->toArray(),
                    'employee_id' => $eventRecord->employee_id ?? '',
                    'employee_nom' => $eventRecord->employee->name . ' ' . $eventRecord->employee->prenom ?? '',
                    'name' => $eventRecord->name ?? '',
                    'note' => $eventRecord->note ?? '',
                    'status' => $eventRecord->status ?? '',
                    'date_debut' => $eventRecord->date_debut ?? '',
                    'date_fin' => $eventRecord->date_fin ?? '',
                    'created_at' => $eventRecord->created_at,
                    'updated_at' => $eventRecord->updated_at
                ]
            ];

        }


        // Add holidays and absences data
        $holidays = Holiday::where('is_active', true)
        ->whereYear('date_debut', '>=', date('Y'))
        ->get();

        // $absences = Absence::with(['employee', 'chantier', 'tache'])
        //     // ->where('statut', 'approved')
        //     ->get();

        // Add to existing events array
        // foreach ($holidays as $holiday) {
        //     $events[] = [
        //         'id' => 'holiday-' . $holiday->id,
        //         'title' => $holiday->name,
        //         'start' => $holiday->date_debut->format('Y-m-d'),
        //         'end' => $holiday->date_fin->addDay()->format('Y-m-d'),
        //         'allDay' => true,
        //         'display' => 'background',
        //         'backgroundColor' => '#ff9800',
        //         'borderColor' => '#ff9800',
        //         'classNames' => ['holiday-event'],
        //         'extendedProps' => [
        //             'type' => 'holiday',
        //             'holiday_id' => $holiday->id,
        //             'editable' => false
        //         ]
        //     ];
        // }

        // foreach ($absences as $absence) {
        //     $events[] = [
        //         'id' => 'absence-' . $absence->id,
        //         'resourceId' => $absence->employee_id,
        //         'title' => $this->getAbsenceTitle($absence),
        //         'start' => $absence->date_debut,
        //         'end' => (new DateTime($absence->date_fin))->modify('+1 day')->format('Y-m-d'),
        //         'allDay' => true,
        //         'backgroundColor' => $this->getAbsenceColor($absence->type),
        //         'borderColor' => $this->getAbsenceColor($absence->type),
        //         'classNames' => ['absence-event'],
        //         'extendedProps' => [
        //             'type' => 'absence',
        //             'absence_id' => $absence->id,
        //             'absence_type' => $absence->type,
        //             'absence_statut' => $absence->statut,
        //             'absence_reason' => $absence->reason,
        //             'category' => $absence->category,
        //             'editable' => true
        //         ]
        //     ];
        // }

        usort($resources, function($a, $b) {
            return ($a['order'] ?? 999) <=> ($b['order'] ?? 999);
        });


        // \Log::info('Events:', $events);

        return view('employee.dashboard', compact('employees', 'resources', 'events', 'chantiers', 'holidays'));
    }

    private function getAbsenceTitle($absence)
    {
        $titles = [
            'absence' => 'Autre',
            'conge_individuel' => 'Congé payé',
            'conge_collectif' => 'Congé collectif',
            'conge_sans_solde' => 'Congé sans solde',
            'absence_injustifiee' => 'Absence injustifiée',
            'absence_justifiee' => 'Absence justifiée',
            'arret_maladie' => 'Arrêt maladie'
        ];
        
        $title = $titles[$absence->type] ?? 'Autre';
        if ($absence->category && $absence->category !== $title) {
            $title .= ' - ' . $absence->category;
        }
        
        return $title;
    }

    private function getAbsenceColor($type)
    {
        return match($type) {
            'absence' => '#607d8b',           // Blue Grey
            'conge_individuel' => '#4caf50',  // Green
            'conge_collectif' => '#ff9800',   // Orange
            default => '#607d8b'
        };
    }


    public function logout()
    {
        $user = User::find(session('current_employee')->id);
        session()->forget("current_employee");

        return redirect()->route('employee.login')->with('success','Success! Déconnecté avec succès.');
    }
    

    public function editProfile(){
        $user = User::find(session('current_employee')->id);
        return view('employee.edit-profile',compact('user'));
    }
    
    public function updateProfile(Request $request){
        $request->validate([
            'password' => 'required',
        ]);
        $user = User::find(session('current_employee')->id);
        
        if(isset($request->password)){
            $user->password = $request->password;
        }
        $user->save();

        
        return redirect()->route('employee.edit-profile')->with('success','Success! mot de passe modifié avec succès.');
    }

    public function elements()
    {
        $elements = Element::where('quantity', '>', 0)->get();
        return view('employee.elements', compact('elements'));
    }

    public function requestElement(Request $request)
    {

        DB::beginTransaction();

        $user = User::find(session('current_employee')->id);

        $request->validate([
            'element_id' => 'required|exists:elements,id',
            'quantity_requested' => 'required|integer|min:1',
            'reason' => 'nullable|string|max:500'
        ]);

        $element = Element::find($request->element_id);
        
        if ($request->quantity_requested > $element->quantity) {
            return redirect()->back()->with('error', 'Quantité prise supérieure au stock disponible.');
        }

        // Update element quantity
        $element->quantity -= $request->quantity_requested;
        $element->save();
        

        ElementRequest::create([
            'employee_id' => $user->id,
            'element_id' => $request->element_id,
            'quantity_requested' => $request->quantity_requested,
            'reason' => $request->reason,
            'status' => 'approved'
        ]);
        
        DB::commit();

        return redirect()->back()->with('success', 'Element pris avec succès.');
    }

    public function myRequests()
    {
        $user = User::find(session('current_employee')->id);

        $requests = ElementRequest::where('employee_id', $user->id)
                                ->orderBy('created_at', 'desc')
                                ->get();
        return view('employee.my-requests', compact('requests'));
    }


    public function materiels()
    {
        $materiels = Materiel::get();//where('status', 'disponible')->
        return view('employee.materiels', compact('materiels'));
    }

    public function requestMateriel(Request $request)
    {
        // try {
            DB::beginTransaction();

            $user = User::find(session('current_employee')->id);

            $request->validate([
                'materiel_id' => 'required|exists:materiels,id',
                'reason' => 'nullable|string|max:500',
                'date_needed' => 'nullable|date|after_or_equal:today'
            ]);

            $materiel = Materiel::find($request->materiel_id);
            
            if ($materiel->status !== 'disponible') {
                return redirect()->back()->with('error', 'Matériel non disponible.');
            }

            // Update materiel status to indisponible
            $materiel->status = 'indisponible';
            $materiel->save();
            


            $affectation = new MaterielSalarie();
            $affectation->salarie_id = $user->id;
            $affectation->materiel_id = $materiel->id;
            $affectation->date_de_prise = now();
            $affectation->status = "assigned";
            $affectation->save();


            MaterielRequest::create([
                'employee_id' => $user->id,
                'materiel_id' => $request->materiel_id,
                'reason' => $request->reason,
                'date_needed' => $request->date_needed,
                'status' => 'approved'
            ]);

            DB::commit();
            

            return redirect()->back()->with('success', 'Matériel emprunté avec succès.');
            
        // } catch (\Exception $e) {
        //     DB::rollBack();
        //     return redirect()->back()->with('error', 'Une erreur est survenue.');
        // }
    }

    public function myMaterielRequests()
    {
        $user = User::find(session('current_employee')->id);

        $requests = MaterielRequest::where('employee_id', $user->id)
                                ->orderBy('created_at', 'desc')
                                ->get();
        return view('employee.my-materiel-requests', compact('requests'));
    }
    
    




























    private function calculateWorkingDays($startDate, $endDate)
    {
        $start = new DateTime($startDate);
        $end = new DateTime($endDate);
        $interval = new DateInterval('P1D');
        $dateRange = new DatePeriod($start, $interval, $end->modify('+1 day'));

        $workingDays = 0;
        foreach ($dateRange as $date) {
            // Skip weekends (6 = Saturday, 0 = Sunday)
            if ($date->format('N') < 6) {
                $workingDays++;
            }
        }
        return $workingDays;
    }

    public function getChantierLastColor(Request $request)
    {
        try {
            $chantierId = $request->input('chantier_id');
            
            // Find the most recent event for this chantier
            $lastEvent = Event::where('chantier_id', $chantierId)
                ->whereNotNull('event_color')
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($lastEvent && $lastEvent->event_color) {
                return response()->json([
                    'success' => true,
                    'color' => $lastEvent->event_color
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'No previous color found for this chantier'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching chantier color: ' . $e->getMessage()
            ], 500);
        }
    }


    public function reorderEmployees(Request $request)
    {
        try {
            $draggedEmployeeId = $request->dragged_employee_id;
            $targetEmployeeId = $request->target_employee_id;
            
            \Log::info('Reorder request:', [
                'dragged' => $draggedEmployeeId,
                'target' => $targetEmployeeId
            ]);
            
            // Start a database transaction
            DB::beginTransaction();
            
            $draggedEmployee = User::find($draggedEmployeeId);
            $targetEmployee = User::find($targetEmployeeId);
            
            if (!$draggedEmployee || !$targetEmployee) {
                DB::rollback();
                return response()->json([
                    'success' => false,
                    'message' => 'Employé non trouvé'
                ]);
            }
            
            // Get all employees ordered by current display_order
            $allEmployees = User::where('role', 'employee')
                ->orderBy('display_order', 'asc')
                ->orderBy('id', 'asc')
                ->get();
            
            \Log::info('Current order before change:', $allEmployees->pluck('name', 'id')->toArray());
            \Log::info('Current display_order values:', $allEmployees->pluck('display_order', 'name')->toArray());
            
            // Create new order array
            $newOrderedEmployees = [];
            $draggedAdded = false;
            
            foreach ($allEmployees as $employee) {
                // Skip the dragged employee in its original position
                if ($employee->id == $draggedEmployeeId) {
                    continue;
                }
                
                // If this is the target employee, add the dragged employee BEFORE it
                if ($employee->id == $targetEmployeeId && !$draggedAdded) {
                    $newOrderedEmployees[] = $draggedEmployee;
                    $draggedAdded = true;
                }
                
                $newOrderedEmployees[] = $employee;
            }
            
            // If dragged employee wasn't added yet (target was last), add it at the end
            if (!$draggedAdded) {
                $newOrderedEmployees[] = $draggedEmployee;
            }
            
            // Update display_order for all employees with explicit error checking
            $updateResults = [];
            foreach ($newOrderedEmployees as $index => $employee) {
                $newDisplayOrder = ($index + 1) * 10;
                
                $result = DB::table('users')
                    ->where('id', $employee->id)
                    ->update(['display_order' => $newDisplayOrder]);
                
                $updateResults[$employee->name] = [
                    'id' => $employee->id,
                    'old_order' => $employee->display_order,
                    'new_order' => $newDisplayOrder,
                    'update_result' => $result
                ];
            }
            
            \Log::info('Update results:', $updateResults);
            
            // Verify the updates worked
            $verifyEmployees = User::where('role', 'employee')
                ->orderBy('display_order', 'asc')
                ->get();
            
            \Log::info('Order after database update:', $verifyEmployees->pluck('name', 'display_order')->toArray());
            
            // Commit the transaction
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Ordre mis à jour avec succès',
                'new_order' => collect($newOrderedEmployees)->pluck('id')->toArray(),
                'update_results' => $updateResults
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('Reorder error:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour: ' . $e->getMessage()
            ]);
        }
    }

    public function storeAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'type' => 'required|in:absence,conge_individuel,conge_collectif',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'category' => 'nullable|string',
            'reason' => 'nullable|string',
            'statut' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::create([
                'employee_id' => $request->employee_id,
                'type' => $request->type,
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'category' => 'Autre',
                'reason' => $request->reason,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin),
                'statut' => $request->statut
                // 'statut' => $request->type === 'conge_collectif' ? 'approved' : 'pending'
            ]);

            DB::commit();

            // Return data in timeline format
            return response()->json([
                'success' => true,
                'absence' => [
                    'id' => $absence->id,
                    'employee_id' => $absence->employee_id,
                    'type' => $absence->type,
                    'date_debut' => $absence->date_debut,
                    'date_fin' => $absence->date_fin,
                    'category' => $absence->category,
                    'reason' => $absence->reason,
                    'statut' => $absence->statut
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la création: ' . $e->getMessage()
            ], 500);
        }
    }

    public function moveAbsence(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'new_start_date' => 'required|date',
            'new_end_date' => 'required|date|after_or_equal:new_start_date',
            'new_employee_id' => 'required|exists:users,id'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $absence->update([
                'employee_id' => $request->new_employee_id,
                'date_debut' => $request->new_start_date,
                'date_fin' => $request->new_end_date,
                'jours_absence' => $this->calculateWorkingDays($request->new_start_date, $request->new_end_date)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Absence déplacée avec succès'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du déplacement: ' . $e->getMessage()
            ], 500);
        }
    }



    public function resizeAbsence(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $absence->update([
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Durée modifiée avec succès',
                'absence' => $absence->fresh()
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error resizing absence:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du redimensionnement: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id',
            'employee_id' => 'required|exists:users,id',
            'date_debut' => 'required|date',
            'date_fin' => 'required|date|after_or_equal:date_debut',
            'type' => 'nullable|in:absence,conge_individuel,conge_collectif',
            'category' => 'nullable|string',
            'reason' => 'nullable|string',
            'statut' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $absence = Absence::findOrFail($request->absence_id);
            
            $updateData = [
                'employee_id' => $request->employee_id,
                'date_debut' => $request->date_debut,
                'date_fin' => $request->date_fin,
                'jours_absence' => $this->calculateWorkingDays($request->date_debut, $request->date_fin)
            ];

            if ($request->has('type')) $updateData['type'] = $request->type;
            if ($request->has('reason')) $updateData['reason'] = $request->reason;
            if ($request->has('statut')) $updateData['statut'] = $request->statut;

            $absence->update($updateData);
            
            DB::commit();

            return response()->json([
                'success' => true,
                'absence' => $absence->fresh()
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la mise à jour: ' . $e->getMessage()
            ], 500);
        }
    }

    public function deleteAbsenceFromTimeline(Request $request)
    {
        $request->validate([
            'absence_id' => 'required|exists:absences,id'
        ]);

        try {
            DB::beginTransaction();
            
            $absence = Absence::findOrFail($request->absence_id);
            $absence->delete();
            
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Absence supprimée avec succès'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la suppression: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateEventEmployee(Request $request)
    {
        try {
            \Log::info('Update Event Request:', $request->all());

            // Validate the request
            $request->validate([
                'event_id' => 'required|exists:events,id',
                'event_name' => 'required|string|max:255',
                'event_start_date' => 'required|date',
                'event_end_date' => 'required|date|after_or_equal:event_start_date',
                'edit-chantier' => 'nullable|exists:chantiers,id',
                'edit-chantier-date_de_debut' => 'nullable|date',
                'edit-chantier-date_de_fin' => 'nullable|date|after_or_equal:edit-chantier-date_de_debut',
                'employee_id' => 'required|exists:users,id',
            ]);

            // Begin transaction
            DB::beginTransaction();

            // Find the event
            $event = Event::findOrFail($request->input('event_id'));
            
            // Store the original color and chantier_id to check if they changed
            $originalColor = $event->event_color;
            $originalChantierId = $event->chantier_id;
            
            if($request->input('edit-chantier')){
                
                // Update the chantier dates if needed
                $chantier = Chantier::findOrFail($request->input('edit-chantier'));
                if($chantier){
                    $chantier->date_de_debut = $request->input('edit-chantier-date_de_debut');
                    $chantier->date_de_fin = $request->input('edit-chantier-date_de_fin');
                    // $chantier->save();

                    $event->chantier_id = $request->input('edit-chantier');
                }
            }

            // Update the event
            $event->name = $request->input('event_name');
            $event->employee_id = $request->input('employee_id');
            $event->date_debut = $request->input('event_start_date');
            $event->date_fin = $request->input('event_end_date');
            $event->note = $request->input('event_note');
            $event->event_color = $request->input('event_color', $event->event_color);
            $event->save();

            // Check if color changed and event is linked to a chantier
            $newColor = $request->input('event_color');
            if ($event->chantier_id && $newColor && $newColor !== $originalColor) {
                // Update all other events for this chantier to use the new color
                Event::where('chantier_id', $event->chantier_id)
                    ->where('id', '!=', $event->id) // Exclude the current event we just updated
                    ->update(['event_color' => $newColor]);
                
                \Log::info('Updated colors for existing events in chantier: ' . $event->chantier_id . ' to color: ' . $newColor);
            }

            // Handle taches
            if ($request->has('taches')) {
                // Get existing tache IDs from the event's taches_id column
                $existingTacheIds = json_decode($event->taches_id, true) ?? [];
                $updatedTacheIds = [];

                // Decode the JSON string to an array
                $taches = json_decode($request->taches, true);
                
                // Ensure we have a valid array
                if (!is_array($taches)) {
                    $taches = [];
                }

                // Process each tache from the request
                foreach ($taches as $tacheData) {
                    $tacheData = (array)$tacheData; // Ensure it's an array
                    
                    if (isset($tacheData['id']) && $tacheData['id']) {
                        // Update existing tache
                        $tache = Tache::find($tacheData['id']);
                        if ($tache) {
                            $tache->update([
                                'nom' => $tacheData['nom'],
                                'description' => $tacheData['description'] ?? null,
                                'date_debut' => !empty($tacheData['date_debut']) ? $tacheData['date_debut'] : null,
                                'date_fin' => !empty($tacheData['date_fin']) ? $tacheData['date_fin'] : null,
                                'nb_jours' => $this->calculateWorkingDays($tacheData['date_debut'], $tacheData['date_fin']),
                                'employees' => json_encode([$event->employee_id]),
                                'status' => $tacheData['status'],
                            ]);
                            $updatedTacheIds[] = (string)$tache->id;
                        }
                    } else {
                        // Create new tache
                        $tache = Tache::create([
                            'nom' => $tacheData['nom'],
                            'description' => $tacheData['description'] ?? null,
                            'date_debut' => !empty($tacheData['date_debut']) ? $tacheData['date_debut'] : null,
                            'date_fin' => !empty($tacheData['date_fin']) ? $tacheData['date_fin'] : null,
                            'status' => $tacheData['status'],
                            'chantier_id' => $event->chantier_id,
                            'nb_jours' => $this->calculateWorkingDays($tacheData['date_debut'], $tacheData['date_fin']),
                            'employees' => json_encode([$event->employee_id]),
                        ]);
                        $updatedTacheIds[] = (string)$tache->id;
                    }
                }

                // Update the event's taches_id with the new array
                $event->taches_id = json_encode($updatedTacheIds);
                $event->save();

                // Delete taches that were removed
                $tachesToDelete = array_diff($existingTacheIds, $updatedTacheIds);
                if (!empty($tachesToDelete)) {
                    Tache::whereIn('id', $tachesToDelete)->delete();
                }
            }

            // Commit transaction
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Événement mis à jour avec succès',
                'event' => $event,
                'chantier' => $chantier ?? ''
            ]);

        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollBack();
            
            // Log the error
            \Log::error('Error updating event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Une erreur est survenue lors de la mise à jour de l\'événement: ' . $e->getMessage()
            ], 500);
        }
    }


    public function deleteEventEmployee(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'event_id' => 'required|exists:events,id',
            ]);

            // Log the deletion request
            \Log::info('Delete Event Request:', [
                'event_id' => $request->input('event_id'),
            ]);

            // Begin transaction
            DB::beginTransaction();
            
            // Delete the event
            $event = Event::findOrFail($request->input('event_id'));
            $event->delete();

            // Commit transaction
            DB::commit();

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Evenement supprimé avec succès',
            ]);
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error deleting Evenement:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Une erreur est survenue lors de la suppression de l\'Evenement',
                'errors' => [
                    'error' => [$e->getMessage()]
                ]
            ], 500);
        }
    }

    public function moveEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Move Event Request:', $request->all());
            
            $request->validate([
                'event_id' => 'required|exists:events,id',
                'new_start_date' => 'required|date',
                'new_end_date' => 'required|date|after_or_equal:new_start_date',
                'new_employee_id' => 'required|exists:users,id',
                'old_employee_id' => 'nullable|exists:users,id'
            ]);

            DB::beginTransaction();

            // Find the event
            $event = Event::findOrFail($request->input('event_id'));
            
            \Log::info('BEFORE Update - Event:', [
                'id' => $event->id,
                'name' => $event->name,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin,
                'employee_id' => $event->employee_id
            ]);
            
            // FIXED: Explicitly update each field and force save
            $event->date_debut = $request->input('new_start_date');
            $event->date_fin = $request->input('new_end_date');
            $event->employee_id = $request->input('new_employee_id');
            
            // Force save and check if it actually saved
            $saveResult = $event->save();
            
            \Log::info('Save result:', ['success' => $saveResult]);
            
            // Refresh from database to confirm changes
            $event->refresh();
            
            \Log::info('AFTER Update - Event from DB:', [
                'id' => $event->id,
                'name' => $event->name,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin,
                'employee_id' => $event->employee_id
            ]);
            
            // Also check the actual database record
            $dbCheck = DB::table('events')->where('id', $event->id)->first();
            \Log::info('Direct DB Check:', [
                'date_debut' => $dbCheck->date_debut,
                'date_fin' => $dbCheck->date_fin,
                'employee_id' => $dbCheck->employee_id
            ]);

            // Update associated taches with new employee if any
            if ($event->taches_id) {
                $tachesIds = json_decode($event->taches_id, true) ?? [];
                if (!empty($tachesIds)) {
                    $updateResult = Tache::whereIn('id', $tachesIds)->update([
                        'employees' => json_encode([$request->input('new_employee_id')])
                    ]);
                    \Log::info('Updated taches count:', $updateResult);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Événement déplacé avec succès',
                'event' => [
                    'id' => $event->id,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'employee_id' => $event->employee_id,
                    'name' => $event->name
                ],
                'debug' => [
                    'request_dates' => [
                        'start' => $request->input('new_start_date'),
                        'end' => $request->input('new_end_date')
                    ],
                    'saved_dates' => [
                        'start' => $event->date_debut,
                        'end' => $event->date_fin
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error moving event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du déplacement: ' . $e->getMessage(),
                'debug' => [
                    'request' => $request->all(),
                    'error' => $e->getMessage()
                ]
            ], 500);
        }
    }

    public function resizeEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Resize Event Request:', $request->all());
            
            $request->validate([
                'event_id' => 'required|exists:events,id',
                'new_start_date' => 'required|date',
                'new_end_date' => 'required|date|after_or_equal:new_start_date'
            ]);

            DB::beginTransaction();

            // Find the event
            $event = Event::findOrFail($request->input('event_id'));
            
            \Log::info('BEFORE Resize - Event:', [
                'id' => $event->id,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin
            ]);
            
            // FIXED: Explicitly update fields
            $event->date_debut = $request->input('new_start_date');
            $event->date_fin = $request->input('new_end_date');
            
            // Force save
            $saveResult = $event->save();
            
            \Log::info('Resize save result:', ['success' => $saveResult]);
            
            // Refresh and verify
            $event->refresh();
            
            \Log::info('AFTER Resize - Event from DB:', [
                'id' => $event->id,
                'date_debut' => $event->date_debut,
                'date_fin' => $event->date_fin
            ]);

            // Update associated taches dates if needed
            if ($event->taches_id) {
                $tachesIds = json_decode($event->taches_id, true) ?? [];
                if (!empty($tachesIds)) {
                    $updateCount = 0;
                    foreach ($tachesIds as $tacheId) {
                        $tache = Tache::find($tacheId);
                        if ($tache) {
                            $tache->date_debut = $request->input('new_start_date');
                            $tache->date_fin = $request->input('new_end_date');
                            $tache->nb_jours = $this->calculateWorkingDays(
                                $request->input('new_start_date'), 
                                $request->input('new_end_date')
                            );
                            $tache->save();
                            $updateCount++;
                        }
                    }
                    \Log::info('Updated taches count:', $updateCount);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Durée de l\'événement modifiée avec succès',
                'event' => [
                    'id' => $event->id,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'name' => $event->name
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error resizing event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du redimensionnement: ' . $e->getMessage()
            ], 500);
        }
    }

    public function duplicateEventEmployee(Request $request)
    {
        try {
            // Log incoming request
            \Log::info('Duplicate Event Request:', $request->all());
            
            $request->validate([
                'original_event_id' => 'required|exists:events,id',
                'duplicate_start_date' => 'required|date',
                'duplicate_end_date' => 'required|date|after_or_equal:duplicate_start_date',
                'duplicate_employee_id' => 'required|exists:users,id'
            ]);

            DB::beginTransaction();

            // Find the original event
            $originalEvent = Event::with(['chantier', 'employee'])->findOrFail($request->input('original_event_id'));
            
            \Log::info('Original Event:', [
                'id' => $originalEvent->id,
                'name' => $originalEvent->name,
                'date_debut' => $originalEvent->date_debut,
                'date_fin' => $originalEvent->date_fin,
                'employee_id' => $originalEvent->employee_id
            ]);
            
            // Create new event as a copy
            $newEvent = new Event();
            $newEvent->name = $originalEvent->name; // Frontend will add " (Copie)" to title
            $newEvent->chantier_id = $originalEvent->chantier_id;
            $newEvent->employee_id = $request->input('duplicate_employee_id');
            $newEvent->date_debut = $request->input('duplicate_start_date');
            $newEvent->date_fin = $request->input('duplicate_end_date');
            $newEvent->note = $originalEvent->note;
            $newEvent->event_color = $originalEvent->event_color;
            $newEvent->status = 'planifie'; // Reset status for duplicated event
            
            // Duplicate taches if any
            $newTachesIds = [];
            if ($originalEvent->taches_id) {
                $originalTachesIds = json_decode($originalEvent->taches_id, true) ?? [];
                
                \Log::info('Duplicating taches:', $originalTachesIds);
                
                foreach ($originalTachesIds as $tacheId) {
                    $originalTache = Tache::find($tacheId);
                    if ($originalTache) {
                        $newTache = new Tache();
                        $newTache->chantier_id = $originalTache->chantier_id;
                        $newTache->nom = $originalTache->nom;
                        $newTache->description = $originalTache->description;
                        
                        // Use the new event dates for the duplicated taches
                        $newTache->date_debut = $request->input('duplicate_start_date');
                        $newTache->date_fin = $request->input('duplicate_end_date');
                        $newTache->nb_jours = $this->calculateWorkingDays(
                            $request->input('duplicate_start_date'), 
                            $request->input('duplicate_end_date')
                        );
                        $newTache->status = 'pending'; // Reset status
                        $newTache->employees = json_encode([$request->input('duplicate_employee_id')]);
                        
                        $newTache->save();
                        $newTachesIds[] = $newTache->id;
                    }
                }
            }
            
            $newEvent->taches_id = json_encode($newTachesIds);
            $newEvent->save();
            
            \Log::info('New Event Created:', [
                'id' => $newEvent->id,
                'name' => $newEvent->name,
                'date_debut' => $newEvent->date_debut,
                'date_fin' => $newEvent->date_fin,
                'employee_id' => $newEvent->employee_id,
                'taches_count' => count($newTachesIds)
            ]);

            // Load relationships for response
            $newEvent->load(['chantier', 'employee']);
            
            // Get taches details for response
            $tachesDetails = [];
            if (!empty($newTachesIds)) {
                $taches = Tache::whereIn('id', $newTachesIds)->get();
                $tachesDetails = $taches->map(function($tache) {
                    return [
                        'id' => $tache->id,
                        'nom' => $tache->nom,
                        'description' => $tache->description,
                        'date_debut' => $tache->date_debut,
                        'date_fin' => $tache->date_fin,
                        'nb_jours' => $tache->nb_jours,
                        'status' => $tache->status,
                        'photo' => $tache->photo
                    ];
                })->toArray();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Événement dupliqué avec succès',
                'event' => [
                    'id' => $newEvent->id,
                    'name' => $newEvent->name,
                    'date_debut' => $newEvent->date_debut,
                    'date_fin' => $newEvent->date_fin,
                    'note' => $newEvent->note,
                    'event_color' => $newEvent->event_color,
                    'employee_id' => $newEvent->employee_id,
                    'chantier_id' => $newEvent->chantier_id,
                    'taches_id' => $newEvent->taches_id,
                    'status' => $newEvent->status,
                    'created_at' => $newEvent->created_at,
                    'updated_at' => $newEvent->updated_at,
                    'employee' => $newEvent->employee ? [
                        'id' => $newEvent->employee->id,
                        'name' => $newEvent->employee->name,
                        'prenom' => $newEvent->employee->prenom
                    ] : null
                ],
                'chantier' => $newEvent->chantier ? [
                    'id' => $newEvent->chantier->id,
                    'nom' => $newEvent->chantier->nom,
                    'adresse' => $newEvent->chantier->adresse,
                    'date_de_debut' => $newEvent->chantier->date_de_debut,
                    'date_de_fin' => $newEvent->chantier->date_de_fin,
                    'nb_jours' => $newEvent->chantier->nbjours
                ] : null,
                'taches_details' => $tachesDetails
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            \Log::error('Error duplicating event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la duplication: ' . $e->getMessage(),
                'debug' => [
                    'request' => $request->all(),
                    'error' => $e->getMessage()
                ]
            ], 500);
        }
    }


    public function storeSelectEventEmployee(Request $request)
    {
        try {
            // Log the incoming data for debugging
            \Log::info('Employee Event Data Received:', $request->all());
    
            // Begin transaction
            DB::beginTransaction();
    
            $tachesIds = [];
    
            // Handle chantier creation or selection
            $chantier = null;

            if ($request->has('chantier_type')) {
                if ($request->input('chantier_type') === 'new' && 
                    $request->has('new_chantier') && 
                    is_array($request->input('new_chantier')) &&
                    !empty($request->input('new_chantier.nom'))) {
                    
                    $nom = $request->input('new_chantier.nom');
                    if (!empty($nom)) {
                        // Create new chantier
                        $chantierId = $this->createNewChantier($request->input('new_chantier'));
                        $chantier = Chantier::find($chantierId);
                    }
                } 
                elseif ($request->input('chantier_type') === 'existing' && 
                    $request->has('existing_chantier_id')) {
                    
                    $chantierId = $request->input('existing_chantier_id');
                    if (!empty($chantierId)) {
                        $chantier = Chantier::find($chantierId);
                        if (!$chantier) {
                            throw new \Exception('Chantier introuvable');
                        }
                    }
                }
            }

            // Later in your code, check if $chantier is not null before using it
            if ($chantier && $request->has('taches') && !empty($request->input('taches'))) {
                $tachesIds = $this->createTaches($request->input('taches'), $chantier->id);
            }
    
            // Create the event
            $event = new Event(); // Adjust the model path as needed
            $event->name = $request->input('event_name');
            if ($chantier) {
                $event->chantier_id = $chantier->id;
            }
            $event->taches_id = json_encode($tachesIds); // Store as JSON array
            // $event->employee_id = $request->input('resource_id'); // The selected employee
            $event->employee_id = $request->input('employee_id'); // The selected employee
            $event->date_debut = $request->input('event_start_date');
            $event->date_fin = $request->input('event_end_date');
            $event->note = $request->input('event_note');
            $event->event_color = $request->input('event_color', '#52357B');
            $event->status = 'planifie'; // Default status
            $event->save();

            if ($chantier && $request->input('chantier_type') === 'existing') {
                $newColor = $request->input('event_color', '#52357B');
                
                // Update all existing events for this chantier to use the new color
                Event::where('chantier_id', $chantier->id)
                    ->where('id', '!=', $event->id) // Exclude the current event we just created
                    ->update(['event_color' => $newColor]);
                
                \Log::info('Updated colors for existing events in chantier: ' . $chantier->id . ' to color: ' . $newColor);
            }
    
            // Commit transaction
            DB::commit();
    
            $event->load(['chantier', 'employee']);

            // Return success response with complete event data
            return response()->json([
                'success' => true,
                'message' => 'Événement créé avec succès',
                'event' => [
                    'id' => $event->id,
                    'name' => $event->name,
                    'date_debut' => $event->date_debut,
                    'date_fin' => $event->date_fin,
                    'note' => $event->note,
                    'event_color' => $event->event_color,
                    'employee_id' => $event->employee_id,
                    'chantier_id' => $event->chantier_id,
                    'taches_id' => $event->taches_id,
                    'status' => $event->status,
                    'created_at' => $event->created_at,
                    'updated_at' => $event->updated_at,
                    'employee' => $event->employee ? [
                        'id' => $event->employee->id,
                        'name' => $event->employee->name,
                        'prenom' => $event->employee->prenom
                    ] : null
                ],
                'chantier' => $chantier ? [
                    'id' => $chantier->id,
                    'nom' => $chantier->nom,
                    'adresse' => $chantier->adresse,
                    'date_de_debut' => $chantier->date_de_debut,
                    'date_de_fin' => $chantier->date_de_fin,
                    'nb_jours' => $chantier->nbjours
                ] : null,
                'taches_count' => count($tachesIds)
            ]);
    
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();
            
            // Log the error
            \Log::error('Error creating employee event:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            // Return error response
            return response()->json([
                'success' => false,
                'message' => 'Une erreur est survenue lors de la création de l\'événement'// . $e->getMessage()
            ], 500);
        }
    }
    





}
